/*===========================================================================*\
*                                                                            *
*                              OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
*                                                                            *
\*===========================================================================*/

/*===========================================================================*\
*                                                                            *
*   $Revision$                                                       *
*   $LastChangedBy$                                                *
*   $Date$                     *
*                                                                            *
\*===========================================================================*/

#include "TextureMath.hh"

#include <cmath>

TextureMath::TextureMath(const bool   _abs,
                         const bool   _clamp,
                         const double _clampMin,
                         const double _clampMax,
                         const bool   _repeat,
                         const double _minRepeat,
                         const double _maxRepeat,
                         const bool   _center,
                         const bool   _scale,
                         const double _minimalInput,
                         const double _maximalInput):
        abs_(_abs),
        clamp_(_clamp),
        clampMin_(_clampMin),
        clampMax_(_clampMax),
        repeat_(_repeat),
        repeatMin_(_minRepeat),
        repeatMax_(_maxRepeat),
        center_(_center),
        scale_(_scale),
        minInput_(_minimalInput),
        maxInput_(_maximalInput)
{

}

TextureMath::TextureMath(const TexParameters& _parameters, const double _minimalInput, const double _maximalInput) :
        abs_(_parameters.abs),
        clamp_(_parameters.clamp),
        clampMin_(_parameters.clampMin),
        clampMax_(_parameters.clampMax),
        repeat_(_parameters.repeat),
        repeatMin_(_parameters.repeatMin),
        repeatMax_(_parameters.repeatMax),
        center_(_parameters.center),
        scale_(_parameters.scale),
        minInput_(_minimalInput),
        maxInput_(_maximalInput)
{
}

double TextureMath::transform(const double _input) const
{
  double value = _input;


  // Use absolute value as requested by plugin
  if ( abs_ )
    value = fabs(value);

  // Clamp if requested
  if ( clamp_ ) {
    if ( value > clampMax_ )
      value = clampMax_;
    if (value < clampMin_)
      value = clampMin_;
  }

  // if all texCoords have the same value
  if ( minInput_ == maxInput_){

    if ( ! repeat_ )
      value = 0.0;
    else
      value = maxInput_;

    return value;
  }


  // if the texture should not be repeated, scale to 0..1
  if ( ! repeat_ ) {
    if (! center_ ) {
      if ( scale_) {
        value /=  fabs(maxInput_) + fabs(minInput_);           //scaleFactor is != 0.0 (otherwise _min==_max)
        value -= minInput_/(fabs(maxInput_) + fabs(minInput_));
      }
    } else {
      // the values above zero are mapped to 0.5..1 the negative ones to 0.5..0
      if (value > 0.0) {
        value /= ( maxInput_ * 2.0); //_max >= _value > 0.0
        value += 0.5;
      } else {
        if ( minInput_ == 0.0 ){
          value = 0.0;
        } else {
          value /= ( minInput_ * 2.0);
          value = 0.5 - value;
        }
      }
    }
  } else {
    value -= minInput_;
    value *= (repeatMax_ - repeatMin_) / (maxInput_ - minInput_);
    value += repeatMin_;
  }

  return value;
}
