/*===========================================================================*\
*                                                                            *
*                              OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
*                                                                            *
\*===========================================================================*/

/*===========================================================================*\
*                                                                            *
*   $Revision$                                                       *
*   $LastChangedBy$                                                *
*   $Date$                     *
*                                                                            *
\*===========================================================================*/



#include "CoreWidget.hh"

#include <OpenFlipper/widgets/viewModeWidget/viewModeWidget.hh>
#include <OpenFlipper/widgets/viewModeWidget/viewModeChangeWidget.hh>

//=============================================================================

/// init ViewModes that were loaded via ini-file
void CoreWidget::initViewModes(){
  //init viewMode subMenu
  if (!viewModeMenu_){
    viewModeMenu_ = new QMenu(tr("View Modes"));
    viewModeButton_ = viewMenu_->addMenu(viewModeMenu_);
    viewGroup_ = new QActionGroup(0);
    viewGroup_->setExclusive(true);
    connect( viewGroup_, SIGNAL( triggered( QAction* ) ), this, SLOT( slotSetViewMode( QAction* ) ) );
  }

  viewModeMenu_->clear();

  bool seenCustom = false;

  for (int i=0; i < viewModes_.size(); i++){

    //Add Separator above the custom widgets
    if (viewModes_[i]->custom && !seenCustom){
      viewModeMenu_->addSeparator();
      seenCustom = true;
    }

    //add Action to viewMenu
    QAction* acViewMode = new QAction(viewModes_[i]->name, this);
    acViewMode->setStatusTip(tr("Change ViewMode"));
    viewGroup_->addAction(acViewMode);
    viewModeMenu_->addAction(acViewMode);

    //add Separator after viewMode 'all'
    if (viewModes_[i]->name == "All")
      viewModeMenu_->addSeparator();
  }


  if ( OpenFlipperSettings().value("Core/Gui/TaskSwitcher/Hide",false).toBool() ) {
      viewModeButton_->setVisible(false);
  }
}

void CoreWidget::slotAddViewModeToolboxes(QString _mode, QStringList _usedWidgets){
  slotAddViewModeToolboxes(_mode, false, _usedWidgets);
}

void CoreWidget::slotAddViewModeToolboxes(QString _mode, bool _custom, QStringList _usedWidgets){
  int id = -1;

  // Check if it already exists
  for ( int i = 0 ; i < viewModes_.size(); i++) {
    if ( viewModes_[i]->name == _mode ) {
      id = i;
      break;
    }
  }

  ViewMode* vm = 0;
  if ( id == -1 ) {
    vm         = new ViewMode();
    vm->name   = _mode;
    vm->custom = _custom;
    vm->icon   = "Unknown.png";

    vm->visibleToolbars = QString("Main Toolbar;Viewer Toolbar").split(";");
    vm->visibleContextMenus = QString("ALL_THAT_EXIST").split(" ");

    if (_custom) {
      viewModes_.push_back(vm);
    } else {
      //insert before custom viewModes
      int i = viewModes_.size();
      for (int k=0; k < viewModes_.size(); k++)
        if (viewModes_[k]->custom == true){
          i = k;
          break;
        }
      viewModes_.insert(i,vm);
    }

  } else {
    vm = viewModes_[id];
  }

  vm->visibleToolboxes = _usedWidgets;

  initViewModes();
}

void CoreWidget::slotAddViewModeToolbars(QString _mode, QStringList _usedToolbars) {
  slotAddViewModeToolbars(_mode,false,_usedToolbars);
}

void CoreWidget::slotAddViewModeToolbars(QString _mode, bool _custom, QStringList _usedToolbars) {
  int id = -1;

  // Check if it already exists
  for ( int i = 0 ; i < viewModes_.size(); i++) {
    if ( viewModes_[i]->name == _mode ) {
      id = i;
      break;
    }
  }

  ViewMode* vm = 0;
  if ( id == -1 ) {
    vm         = new ViewMode();
    vm->name   = _mode;
    vm->custom = _custom;
    vm->icon   = "Unknown.png";

    vm->visibleContextMenus = QString("ALL_THAT_EXIST").split(" ");

    if (_custom) {
      viewModes_.push_back(vm);
    } else {
      //insert before custom viewModes
      int i = viewModes_.size();
      for (int k=0; k < viewModes_.size(); k++)
        if (viewModes_[k]->custom == true){
          i = k;
          break;
        }
        viewModes_.insert(i,vm);
    }
  } else {
    vm = viewModes_[id];
  }

  // Always add the viewer Toolbar
  if ( ! _usedToolbars.contains("Viewer Toolbar") )
  _usedToolbars.prepend("Viewer Toolbar");

  // Always add the main Toolbar
  if ( ! _usedToolbars.contains("Main Toolbar") )
    _usedToolbars.prepend("Main Toolbar");

  vm->visibleToolbars = _usedToolbars;

  initViewModes();
}


void CoreWidget::slotAddViewModeContextMenus(QString _mode, QStringList _usedContextMenus){
  slotAddViewModeContextMenus(_mode, false, _usedContextMenus);
}

void CoreWidget::slotAddViewModeContextMenus(QString _mode, bool _custom, QStringList _usedContextMenus){
  int id = -1;

  // Check if it already exists
  for ( int i = 0 ; i < viewModes_.size(); i++) {
    if ( viewModes_[i]->name == _mode ) {
      id = i;
      break;
    }
  }

  ViewMode* vm = 0;
  if ( id == -1 ) {
    vm         = new ViewMode();
    vm->name   = _mode;
    vm->custom = _custom;
    vm->icon   = "Unknown.png";

    vm->visibleToolbars = QString("Main Toolbar;Viewer Toolbar").split(";");

    if (_custom) {
      viewModes_.push_back(vm);
    } else {
      //insert before custom viewModes
      int i = viewModes_.size();
      for (int k=0; k < viewModes_.size(); k++)
        if (viewModes_[k]->custom == true){
          i = k;
          break;
        }
      viewModes_.insert(i,vm);
    }

  } else {
    vm = viewModes_[id];
  }

  vm->visibleContextMenus = _usedContextMenus;

  initViewModes();
}

/// Sets the Icon for a given View Mode
void CoreWidget::slotSetViewModeIcon(QString _mode, QString _iconName) {
  slotSetViewModeIcon(_mode,false,_iconName);
}

/// Sets the Icon for a given View Mode
void CoreWidget::slotSetViewModeIcon(QString _mode, bool _custom, QString _iconName) {

  int id = -1;

  // Check if it already exists
  for ( int i = 0 ; i < viewModes_.size(); i++) {
    if ( viewModes_[i]->name == _mode ) {
      id = i;
      break;
    }
  }

  ViewMode* vm = 0;
  if ( id == -1 ) {
    vm         = new ViewMode();
    vm->name   = _mode;
    vm->custom = _custom;
    vm->icon   = _iconName;

    if (_custom) {
      viewModes_.push_back(vm);
    } else {
      //insert before custom viewModes
      int i = viewModes_.size();
      for (int k=0; k < viewModes_.size(); k++)
        if (viewModes_[k]->custom == true){
          i = k;
          break;
        }
        viewModes_.insert(i,vm);
    }
  } else {
    vm = viewModes_[id];
  }

  vm->icon = _iconName;

  initViewModes();
}

/// Remove a viewMode
void CoreWidget::slotRemoveViewMode(QString _name){
  for (int i=0; i < viewModes_.size(); i++)
    if (viewModes_[i]->name == _name && viewModes_[i]->custom == true){ //remove only userdefined viewModes
      viewModes_.remove(i);
      //remove action from menu
      for (int a=0; a < viewModeMenu_->actions().size(); a++)
        if (viewModeMenu_->actions()[a]->text() == _name){
          QAction* action = viewModeMenu_->actions()[a];
          viewModeMenu_->removeAction(action);
          viewGroup_->removeAction(action);
          delete action;
        }
      break;

    }
}

/// Slot for setting the viewMode from menu
void CoreWidget::slotSetViewMode( QAction* action){
  setViewMode( action->text() );
}

/// Slot for setting the viewMode from menu
void CoreWidget::setViewMode( QString _mode, bool _expandAll ){
  slotChangeView(_mode, QStringList(), QStringList(), QStringList(), _expandAll);
}

void CoreWidget::slotAddViewModeComplete(QString _mode , bool _custom, QStringList _toolboxes, QStringList _toolbars, QStringList _contextmenus) {
  slotAddViewModeToolbars(_mode,_custom,_toolbars);
  slotAddViewModeToolboxes(_mode,_custom,_toolboxes);
  slotAddViewModeContextMenus(_mode,_custom,_contextmenus);
}

/// show dialog for changing ViewMode
void CoreWidget::slotViewModeDialog(){
  //init widget
  static viewModeWidget* widget = 0;
  if ( !widget ){
    widget = new viewModeWidget(viewModes_);
    widget->setWindowIcon( OpenFlipper::Options::OpenFlipperIcon() );
    connect(widget, SIGNAL(changeView(QString, QStringList, QStringList, QStringList)), this, SLOT(slotChangeView(QString, QStringList, QStringList, QStringList)) );
    connect(widget, SIGNAL(saveMode(QString, bool, QStringList, QStringList, QStringList)), this, SLOT(slotAddViewModeComplete(QString, bool, QStringList, QStringList, QStringList)) );
    connect(widget, SIGNAL(removeMode(QString)), this, SLOT(slotRemoveViewMode(QString)) );
  }
  widget->show( OpenFlipper::Options::currentViewMode() );
}

void CoreWidget::closeChangeViewModePopup() {
    QWidget *parent = qobject_cast<QWidget*>(modeChangeWidget->parent());
    if (parent)
        parent->close();
}

/// Slot for Changing visible toolWidgets
void CoreWidget::slotChangeView(QString _mode, QStringList _toolboxWidgets, QStringList _toolbars, QStringList _contextmenus, bool _expandAll ){

  //try to find Widgets if they aren't given
  if (_mode != "" && _toolboxWidgets.size() == 0 && _toolbars.size() == 0)
    for (int i=0; i < viewModes_.size(); i++)
      if (viewModes_[i]->name == _mode) {
        _toolboxWidgets = viewModes_[i]->visibleToolboxes;
        _toolbars       = viewModes_[i]->visibleToolbars;
        _contextmenus	  = viewModes_[i]->visibleContextMenus;
      }


  // Remove all toolbox entries if the view has changed
  if (_mode != OpenFlipper::Options::currentViewMode()) {
    toolBox_->saveViewModeState(OpenFlipper::Options::currentViewMode());
    toolBox_->clear();
  }

  //find all widgets that should be visible
  for (int i=0; i < _toolboxWidgets.size(); i++) {
    for (uint p=0; p < plugins_.size(); p++){
      for ( uint j = 0 ; j < plugins_[p].toolboxWidgets.size(); ++j )
        if (_toolboxWidgets[i] == plugins_[p].toolboxWidgets[j].first ) {

          bool skip = false;
          if (toolBox_->plugins().contains(plugins_[p].plugin)) {
            // account for the case, where a plugin can have several
            // toolboxes, for example 'Scripting'
            if (toolBox_->names().contains(_toolboxWidgets[i]))
              skip = true;
          }

          // only add items that have not been added yet
          if (!skip) {
            toolBox_->addItem (plugins_[p].plugin, plugins_[p].toolboxWidgets[j].second, plugins_[p].toolboxWidgets[j].first, plugins_[p].toolboxIcons[j], plugins_[p].headerAreaWidgets[j].second );

            // move item to the correct position
            if (i < toolBox_->lastPos_) {
              toolBox_->moveItemToPosition(plugins_[p].plugin, _toolboxWidgets[i], i);
            } else
              toolBox_->lastPos_ = i;

            // check if we have to restore the state
            // of toolboxes added via scripts
            if (plugins_[p].name == "Scripting") {

              QFile statesFile(OpenFlipper::Options::configDirStr()  + OpenFlipper::Options::dirSeparator() + "WindowStates.dat");

              if (statesFile.exists() ) {
                QSettings windowStates(OpenFlipper::Options::configDirStr()  + OpenFlipper::Options::dirSeparator() + "WindowStates.dat", QSettings::IniFormat);


                windowStates.beginGroup ("Core");
                windowStates.beginGroup("SideArea");
                windowStates.beginGroup(_toolboxWidgets[i]);
                bool active = windowStates.value ("Active", false).toBool();
                windowStates.endGroup();
                windowStates.endGroup();
                windowStates.endGroup();

                toolBox_->setElementActive(_toolboxWidgets[i], active);
              }
            }
          }
        }
    }
  }

  if (_mode != OpenFlipper::Options::currentViewMode()) {
    toolBox_->restoreViewModeState(_mode);
  }

  if (_expandAll)
    toolBox_->expandAll();

  if ( ! OpenFlipperSettings().value("Core/Gui/Toolbar/hidden",false).toBool())
  {
    //find all Toolbars that should be visible and hide the others
    for (uint p=0; p < plugins_.size(); p++)
      for ( uint j = 0 ; j < plugins_[p].toolbars.size(); ++j )
        if (_toolbars.contains( plugins_[p].toolbars[j].first ) )
          plugins_[p].toolbars[j].second->show();
        else
          plugins_[p].toolbars[j].second->hide();


    // Check the Main Toolbar:
    if ( _toolbars.contains(tr("Main Toolbar")) )
      mainToolbar_->show();
    else
      mainToolbar_->hide();

    // Check the Main Toolbar:
    if ( _toolbars.contains(tr("Viewer Toolbar")) )
      viewerToolbar_->show();
    else
      viewerToolbar_->hide();
  }


  if (_mode != "")
    OpenFlipper::Options::currentViewMode(_mode);

}

void CoreWidget::moveToolBoxToTop(QString _name) {

  toolBox_->moveItemToPosition(_name, 0);
}

void CoreWidget::moveToolBoxToBottom(QString _name) {

  toolBox_->moveItemToPosition(_name, toolBox_->getNumberOfWidgets()-1);
}

void CoreWidget::stereoButtonContextMenu(const QPoint& _pos) {

    // Grey out OpenGL stereo mode option if not available
    if(!OpenFlipper::Options::glStereo()) {
        stereoSettingsWidget_->stereoOpengl->setDisabled(true);
    } else {
        stereoSettingsWidget_->stereoOpengl->setChecked(OpenFlipper::Options::stereoMode() == OpenFlipper::Options::OpenGL);
    }

    // Set values
    stereoSettingsWidget_->stereoAnaglyph->setChecked(OpenFlipper::Options::stereoMode() == OpenFlipper::Options::AnaglyphRedCyan);
    stereoSettingsWidget_->stereoCustomAnaglyph->setChecked(OpenFlipper::Options::stereoMode() == OpenFlipper::Options::AnaglyphCustom);

    stereoSettingsWidget_->eyeDistance->setValue( OpenFlipperSettings().value("Core/Stereo/EyeDistance").toDouble() );
    stereoSettingsWidget_->focalDistance->setValue( OpenFlipperSettings().value("Core/Stereo/FocalDistance").toDouble() * 1000);

    // Move widget to the position of the cursor
    stereoSettingsWidget_->move(stereoButton_->mapToGlobal(_pos) - QPoint((int)(stereoSettingsWidget_->width()/2), 0));
    // Show widget
    stereoSettingsWidget_->show();
}

void CoreWidget::slotApplyStereoSettings(int /*_tmpParam*/) {

    // Update values
    if (stereoSettingsWidget_->stereoCustomAnaglyph->isChecked()) {
        // Update option entry
        OpenFlipper::Options::stereoMode(OpenFlipper::Options::AnaglyphCustom);
        // Show right stacked widget
        stereoSettingsWidget_->stackedWidget->setCurrentIndex(0);
    } else if (stereoSettingsWidget_->stereoAnaglyph->isChecked()) {
        OpenFlipper::Options::stereoMode(OpenFlipper::Options::AnaglyphRedCyan);
        stereoSettingsWidget_->stackedWidget->setCurrentIndex(0);
    } else {
        OpenFlipper::Options::stereoMode(OpenFlipper::Options::OpenGL);
        stereoSettingsWidget_->stackedWidget->setCurrentIndex(0);
    }

    // Save everything
    OpenFlipperSettings().setValue("Core/Stereo/EyeDistance",stereoSettingsWidget_->eyeDistance->value());
    OpenFlipperSettings().setValue("Core/Stereo/FocalDistance",double(stereoSettingsWidget_->focalDistance->value()/1000.0));

    // Update all views
    for (unsigned int i = 0 ; i < OpenFlipper::Options::examinerWidgets() ; ++i) {
        examiner_widgets_[i]->updateGL();
    }
}

void CoreWidget::slotUpdateExaminer(unsigned _id)
{
  examiner_widgets_[_id]->updateGL();
}
