/*===========================================================================*\
*                                                                            *
*                              OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
*                                                                            *
\*===========================================================================*/






#ifndef TOOLS_INIFILE_HH
#define TOOLS_INIFILE_HH

#include <OpenFlipper/common/GlobalDefines.hh>
#include <QString>
#include <QFile>
#include <map>
#include <vector>


//! Class for the handling of simple configuration files.
/*! The file format used by this class is similar to that of MS
  Windows INI files. It is possible to divide a file into sections
  which are denoted by brackets (e.g. <tt>[Section]</tt>). All values
  are stored in the form <tt>\<key\> = \<value\></tt> and are assigned
  to a section. The INIFile class is able to store values of the types
  string, double, signed and unsigned int and bool.

  The usual way to work with this class consists of the following steps:

  - Open an existing or create a new INI file with a call to connect()

  - Check for the existence of entries (entry_exists()), access stored
    values (get_entry()) or add new entries (add_entry()). Note that
    the function add_section() is obsolete and only kept for backward
    compatibility. A new section is now implicitly generated by
    specifying it in a call to add_entry().

  - disconnect() from the INI file. Only now are all entries and
    sections written to the file.

  To be able to use the INIFile class, an object is created. After
  that the object is connected to a given filename. This means that,
  if the file exists, it is opened and the content is parsed and
  stored in memory. The second (boolean) parameter to the connect()
  function indicates that the file should be created if it does not
  exists.  \until }

  Now a new section named <tt>Main</tt> is generated and several
  entries are added. Note that the INI file is not written to disc
  until the disconnect() function is called. All additions and
  modifications are only performed on the stored data in memory.
  \until iniFile.disconnect();

  In the second part of this example the just created INI file is
  opened (connected) again and the created entries are read and printed
  to <tt>stdout</tt>.
  \until }
  \until }
  \until }

*/
class DLLEXPORT INIFile
{
public :

    //! Default constructor
    INIFile();

    //! Destructor
    ~INIFile();

    //! Connect INIFile object with given filename
    bool connect( const QString& name,
		  const bool create );

    //! Remove connection of this object to a file
    void disconnect();

    //! Check if object is connected to  file
    bool is_connected() const { return mf_isConnected; }

    //! Access to name of connected file
    const QString& name() const { return m_filename; }


    // ---------------------------------------- check if something is there

    /*! \name Existence checking */
    //@{

    //! Check if given section exists in the current INI file
    bool section_exists(const QString & _section) const;

    //! Check if given entry esists in the current INI file
    bool entry_exists(const QString & _section, const QString & _key) const;

    //! Same as entry_exists() (for backward compatibility)
    bool entry_in_section(const QString & _section,
			  const QString & _key) const
	{ return entry_exists( _section, _key ); }

    // ---------------------------------------- section manipulation

    /*! \name Addition of entities */
    //@{

    //! Addition of a section
    /* \deprecated This function is not needed any more. New sections are
       implicitly created upon calls to add_entry().
    */
    void add_section(const QString & _sectionname);

    //! Addition / modification of a string entry
    void add_entry( const QString & _section,
		    const QString & _key,
		    const QString & _value );

    //! Addition / modification of a string entry, given as char array
    void add_entry( const QString & _section,
		  const QString & _key,
		  const char * _value )
	{ add_entry( _section, _key, QString(_value) ); }

    //! Addition / modification of a double entry
    void add_entry( const QString & _section,
		    const QString & _key,
		    const double & _value);

    //! Addition / modification of a double entry
    void add_entry( const QString & _section,
		    const QString & _key,
		    const float & _value);

    //! Addition / modification of an int entry
    void add_entry( const QString & _section,
		    const QString & _key ,
		    const int & _value);

    //! Addition / modification of an int entry
    void add_entry( const QString & _section,
		    const QString & _key ,
		    const unsigned int & _value);

    //! Addition / modification of a boolean entry
    void add_entry( const QString & _section,
		    const QString & _key ,
		    const bool & _value);

    //! Addition / modification of a float vector
    void add_entry( const QString & _section,
		    const QString & _key ,
		    const std::vector<float> & _value);

    //! Addition / modification of a double vector
    void add_entry( const QString & _section,
                    const QString & _key ,
                    const std::vector<double> & _value);
                    
    //! Addition / modification of a bool vector
    void add_entry( const QString & _section,
                    const QString & _key ,
                    const std::vector<bool> & _value);

    //! Addition of a Vec_n_something
    template < typename VectorT >
    void add_entryVec ( const QString & _section,
                        const QString & _key,
                        const VectorT  & _value);

    //! Addition of a vector of Vec_n_something
    template < typename VectorT >
    void add_entryVec ( const QString & _section,
                        const QString & _key,
                        const std::vector< VectorT > & _value);

    //! Addition / modification of an int vector
    void add_entry( const QString & _section,
          const QString & _key ,
          const std::vector<int> & _value);

    //! Addition / modification of an int vector
    void add_entry( const QString & _section,
          const QString & _key ,
          const std::vector<QString> & _value);

    //! Addition / modification of an int vector
    void add_entry( const QString & _section,
          const QString & _key ,
          const QStringList & _value);


    //@}


    // -------------------- delete

    /*! \name Deletion of entities */
    //@{

    //! Deletion of an entry
    void delete_entry( const QString & _section, const QString & _key);

    //! Deletion of an entire section
    void delete_section( const QString & _sectionname );

    //@}



    // -------------------- retrieval

    /*! \name Access to stored entries */
    //@{

    //! Access to a string entry
    bool get_entry( QString & _val,
		    const QString & _section,
		    const QString & _key ) const;

    //! Access to a floating point entry, stored as double value
    bool get_entry( double & _val,
		    const QString & _section,
		    const QString & _key ) const;

    //! Access to a floating point entry, stored as float value
    bool get_entry( float & _val,
		    const QString & _section,
		    const QString & _key ) const;

    //! Access to an int entry
    bool get_entry( int & _val,
		    const QString & _section,
		    const QString & _key ) const;

    //! Access to an unsigned int entry (which is actually stored as int!!!)
    bool get_entry( unsigned int & _val,
		    const QString & _section,
		    const QString & _key ) const;

    //! Access to a boolean entry
    bool get_entry( bool & _val,
		    const QString & _section,
		    const QString & _key) const;

    //! Access to a float vector
    bool get_entry( std::vector<float> & _val,
		    const QString & _section,
		    const QString & _key) const;

    //! Access to a double vector
    bool get_entry( std::vector<double> & _val,
                    const QString & _section,
                    const QString & _key) const;
                    
    //! Access to a bool vector
    bool get_entry( std::vector<bool> & _val,
                    const QString & _section,
                    const QString & _key) const;

    //! Access to an int vector
    bool get_entry( std::vector<int> & _val,
          const QString & _section,
          const QString & _key) const;

    //! Access to a string vector
    bool get_entry( std::vector<QString> & _val,
          const QString & _section,
          const QString & _key ) const;

    //! Access to a stringList
    bool get_entry( QStringList & _val,
          const QString & _section,
          const QString & _key ) const;

    //! Get a Vec_n_d (double)
    template < typename VectorT >
    bool get_entryVecd ( VectorT & _val ,
                         const QString & _section,
                         const QString & _key ) const;

        //! Get a Vec_n_i (int)
    template < typename VectorT >
    bool get_entryVecf ( VectorT & _val ,
                         const QString & _section,
                         const QString & _key ) const;

    //! Get a Vec_n_i (int)
    template < typename VectorT >
    bool get_entryVeci ( VectorT & _val ,
                         const QString & _section,
                         const QString & _key ) const;

    //! Get a Vec_n_d
    template < typename VectorT >
    bool get_entryVecd (std::vector< VectorT > & _val ,
                       const QString & _section,
                       const QString & _key ) const;

    //! Get a Vec_n_f
    template < typename VectorT >
        bool get_entryVecf (std::vector< VectorT > & _val ,
                           const QString & _section,
                           const QString & _key ) const;

    //! Get a Vec_n_i
    template < typename VectorT >
    bool get_entryVeci (std::vector< VectorT > & _val ,
                        const QString & _section,
                        const QString & _key ) const;


private: // data


    //! Type for map of contained entries
    typedef std::map< QString, QString > EntryMap;


    //! Type for map of contained sections
    typedef std::map< QString, EntryMap > SectionMap;


    /////////////////////////////////////////////////////////////////
    /////////////////////////////////////////////////////////////////

    //! Read content of an INI file
    bool parseFile( QFile & _inputStream );

    //! Write data to file we are currently connected to
    bool writeFile( void );

    //! Name of current INI file
    QString m_filename;

    //! Flag: this object is connected to an INI file
    bool mf_isConnected;


    //! Stored data of an INI file
    SectionMap m_iniData;
};


#if defined(INCLUDE_TEMPLATES) && !defined(INIFILE_C)
#define INIFILE_TEMPLATES
#include "INIFileT_impl.hh"
#endif

#endif
