/*===========================================================================*\
*                                                                            *
*                              OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
*                                                                            *
\*===========================================================================*/

#pragma once

#include <ACG/Scenegraph/DrawModes.hh>
#include <OpenFlipper/common/ViewerProperties.hh>

/** \file RenderInterface.hh
*
* Interface for adding global rendering functions. \ref renderInterfacePage
*/

/** \brief Interface to add additional rendering functions from within plugins.
 *
 * \ref renderInterfacePage "Detailed description"
 * \n
 * \n
 *
 * Interface for adding and controlling rendering functions from a plugin.\n
*/
class RenderInterface {

  public:

    /// Destructor
    virtual ~RenderInterface() {};

  private slots:

    /** \brief announce required draw modes
     *
     * This function is called by the core to get a list of draw modes that are supported
     * by this renderer. If such a draw mode is set and the currently active
     * renderer does not support the given mode, the core will switch to
     * this renderer automatically. If there are multiple renderers capable of rendering
     * the drawmode(s), the core will ask the user for its preference.
     *
     * @param _mode Combined list of drawmodes
     */
    virtual void supportedDrawModes(ACG::SceneGraph::DrawModes::DrawMode& _mode) = 0;

    /** \brief announce name for the rendering function
     *
     *
     * @return Name displayed for the rendering function
     */
    virtual QString rendererName() = 0;

    /** \brief Check OpenGL capabilities
     *
     * This function has to be implemented and checks, if all required OpenGL extensions are available.
     * If this is not the case, the plugin will be refused by the core to avoid crashes due to insufficient
     * OpenGL support.
     *
     * You can get the version information in the following way:
     *
     * \code
     *
     * // Get version and check
     * QGLFormat::OpenGLVersionFlags flags = QGLFormat::openGLVersionFlags();
     * if ( ! flags.testFlag(QGLFormat::OpenGL_Version_2_1) )
     *   return QString("Insufficient OpenGL Version! OpenGL 2.1 or higher required");
     *
     * //Get OpenGL extensions
     * QString glExtensions = QString((const char*)glGetString(GL_EXTENSIONS));
     *
     * // Collect missing extension
     * QString missing = "";
     *
     * if ( !glExtensions.contains("GL_ARB_vertex_program") )
     *   missing += "Missing Extension GL_ARB_vertex_program\n";
     *
     * return missing;
     * \endcode
     *
     * @return Return an empty string if everything is fine, otherwise return, what features are missing.
     */
    virtual QString checkOpenGL() = 0;

  public slots:

    /** \brief rendering function
     *
     * Implement this function to render the scene. You have to traverse the
     * scenegraph in this function. The glContext is made current before this
     * call.
     * If you need to initialize something,you can do it within this function.
     * gl and glew are already setUp here.
     */
    virtual void render(ACG::GLState* _glState, Viewer::ViewerProperties& _properties) {};

    /** \brief Return options menu
     *
     * If you want an options Menu or menu entry, you can return your action here.
     * It will be shown on top of the renderer list in the options menu.
     *
     *
     * @return Action for a menu or menu entry
     */
    virtual QAction* optionsAction() { return 0; };

    /** \brief Return a qstring of the current render objects
     *
     * In this function your plugin should return a list of the render objects
     * (possibly with additional information). It can be used  for debugging purposes.
     *
     * The IRenderer class already has a function called dumpCurrentRenderObjectsToString()
     * which can be used to implement this function.
     *
     * @param _outputShaderInfo Output shader code along with the render objects
     * @return string of render objects
     */
    virtual QString renderObjectsInfo(bool _outputShaderInfo) { return QString("Render object inforation not implemented in this plugin"); };

    
    /** \brief Reload any renderer specific shaders
     *
     * This function allows the plugin to reload any renderer specific shaders.
     */
    virtual void reloadShaders() {  };


};

/** \page renderInterfacePage Render Interface
\image html RenderInterface.png
\n

The RenderInterface can be used by plugins to add special rendering functions to OpenFlipper.
You can create a dedicated rendering function for a drawmode. E.g. when you need to render 4
passes with different shaders you can do that in your rendering function.


Example Code for functions:
\code
 TODO
\endcode


To use the RenderInterface:
<ul>
<li> include RenderInterface.hh in your plugins header file
<li> derive your plugin from the class RenderInterface
<li> add Q_INTERFACES(RenderInterface) to your plugin class
<li> And add the signals or slots you want to use to your plugin class (You don't need to implement all of them)
</ul>

*/


Q_DECLARE_INTERFACE(RenderInterface,"OpenFlipper.RenderInterface/1.0")

