/*===========================================================================*\
*                                                                            *
*                              OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
*                                                                            *
\*===========================================================================*/


//=============================================================================
//
//  Types
//
//=============================================================================

/**
 * \file BaseObjectData.hh
 * This File contains the Basic object class for all Objects which show content vi
 * the Scenegraph.
 */

#pragma once


//== INCLUDES =================================================================

#include <OpenFlipper/common/GlobalDefines.hh>
#include <OpenFlipper/common/BaseObject.hh>
#include <vector>
#include <ACG/Scenegraph/PickTarget.hh>
#include <ACG/Scenegraph/DrawModes.hh>

//== FORWARD Declarations to avoid gl qt collisions =================================================================

namespace ACG {
namespace SceneGraph {

class MaterialNode;
class QtTranslationManipulatorNode;
class SeparatorNode;
class BaseNode;
class BoundingBoxNode;
class StencilRefNode;
class ShaderNode;

}
}


//== TYPEDEFS FOR SCENEGRAPH ===============================================
/// Materialnode
typedef ACG::SceneGraph::MaterialNode                     MaterialNode;
/// ManipulatorNode
typedef ACG::SceneGraph::QtTranslationManipulatorNode     QtTranslationManipulatorNode;
/// Seperator Node
typedef ACG::SceneGraph::SeparatorNode                    SeparatorNode;
/// Base Node
typedef ACG::SceneGraph::BaseNode                         BaseNode;
/// Bounding box Node
typedef ACG::SceneGraph::BoundingBoxNode                  BoundingBoxNode;
/// Stencil reference Node
typedef ACG::SceneGraph::StencilRefNode                   StencilRefNode;

//== CLASS DEFINITION =========================================================

/**
 * This is the basic Data class providing the functions common to all objects which show Objects in the SceneGraph
 */
class DLLEXPORT BaseObjectData : public BaseObject
{

  friend class Core;

  Q_OBJECT
  
  public:

    /** \brief copy constructor
     *
     *  Create new basic scenegraph nodes for this object
     */
    BaseObjectData(const BaseObjectData& _object);

    /// constructor
    BaseObjectData();

    ///destructor
    virtual ~BaseObjectData();

  private:
    /** This function creates the basic scenegraph nodes */
    void initializeScenegraphNodes();

  /** @} */

  //===========================================================================
  /** @name Data
   * @{ */
  //===========================================================================

  public:
    /** Clean all data structures of the object
      *
      * */
    virtual void cleanup();


  /** @} */

    virtual void setName(QString _name );

  //===========================================================================
  /** @name Object visualization
   * @{ */
  //===========================================================================

  public :
    /// Sets the whole Scenegraph subtree of this node to visible
    virtual void show();

    /// Sets the whole Scenegraph subtree of this node to invisible
    virtual void hide();

    /// return visiblity
    virtual bool visible();

    /// Sets visiblity of the whole Scenegraph subtree of this node
    virtual void visible(bool _visible);

    /** get the base node of this object (Use this node to add custom Nodes to the Object
     * which should not be transformed with the manipulator of the Object)
     */
    SeparatorNode* baseNode();

    /** \brief Check if the given node is owned by this object
    *
    * You can overload this function and return true, if your object generated the given node. 
    * Don't forget to call this baseclass function on overload!
    */
    virtual bool hasNode(BaseNode* _node);
    
    /** get the primary node of this object (Use this node to change drawModes)
     */
    virtual BaseNode* primaryNode();
    
    /** get the ManipulatorNode node of this object (Use this node to add custom Nodes to the Object
     * which should be transformed with the manipulator of the Object)
     */
    QtTranslationManipulatorNode* manipulatorNode();

    /** Return pointer to the shader node
     * If you want to support shaders, you have to add a shader node into your scenegraph structure
     * above your object to be rendered. If you do not have a shader, just ignore this function
     * and it will return a 0 pointer.
     */
    virtual ACG::SceneGraph::ShaderNode* shaderNode();

    /// get a pointer to the materialnode
    MaterialNode* materialNode();
    
    /// get a const pointer to the materialnode
    const MaterialNode* materialNode() const;
    
    /// get a pointer to the bounding box node
    BoundingBoxNode* boundingBoxNode();

    /// get a pointer to the stencil reference node
    StencilRefNode* stencilRefNode();

    /// Check if the manipulator has been placed
    bool manipPlaced();

    /// set the manipulator place status
    void manipPlaced( bool _placed );

    /// get the bounding box of the object
    void getBoundingBox(ACG::Vec3d& bbmin, ACG::Vec3d& bbmax);
    
    /** \brief Set the draw mode for the object
     * @param _mode  The draw mode that should be active for this object
     * @param _force If true, the mode is set ignoring if its supported by the node and its subnodes
     */
    void setObjectDrawMode(const ACG::SceneGraph::DrawModes::DrawMode& _mode, const bool& _force = false);

  private :


    bool manipPlaced_;

    /// rootNode of global Scenegraph structure
    SeparatorNode*    rootNode_;

    /// Separator at top of Scenegraph structure used for this Object
    SeparatorNode*    separatorNode_;

    /// Manipulator used for this Object
    QtTranslationManipulatorNode*  manipulatorNode_;

    /// Scenegraph Material Node for the object
    MaterialNode*    materialNode_;

    /// Bounding box node for the object
    BoundingBoxNode * boundingBoxNode_;

    /// Stencil reference node for the object
    StencilRefNode* stencilRefNode_;

  /** @} */


  //===========================================================================
  /** @name Picking
   * @{ */
  //===========================================================================

  public :
    /// detect if the node has been picked
    virtual bool picked( uint _node_idx );

    /** Enable or disable picking for this Node
     * The basic function defined here does nothing.
     * It has to be reimplemented in the derived class
     */
    virtual void enablePicking( bool _enable );

    /** Check if picking is enabled for this Node
     * This function will return true unless the derived class overwrites
     * this function.
     */
    virtual bool pickingEnabled();

    /** \brief Refine picking
     *
     * Reimplement this function, if the object type can refine picking. E.g.
     * the standard picking will rely on the z-buffer resolution, while a mesh
     * can intersect a ray with the triangle and refine the depth via
     * the exact intersection point.
     *
     * @param _pickTarget Current picking target (Faces, all,...)
     * @param _original hitpoint
     * @param _eyePos current picking ray start position
     * @param _dir Ray direction when picking
     * @param _targetIdx Entity id hit (e.g. face index for meshes
     *
     */
    virtual ACG::Vec3d refinePick(ACG::SceneGraph::PickTarget _pickTarget,
                                  const ACG::Vec3d _hitPoint,
                                  const ACG::Vec3d _start ,
                                  const ACG::Vec3d _dir,
                                  const unsigned int _targetIdx );

  /** @} */

  //===========================================================================
  /** @name Update handling
   * @{ */
  //===========================================================================

  protected:
    /** \brief  This function is called to update the object
    *
    * If the object changes, the core will call this function. Normally this will update
    * the corresponding scenegraph nodes or trigger other data handling which has to be done
    * when the object changes.
    *
    * \note Do not call this function yourself to avoid unnecessary overhead(the core will call it via the type plugins when it is required)
    */
    virtual void update(UpdateType _type = UPDATE_ALL );

  /** @} */

  //===========================================================================
  /** @name Additional nodes
   *
   *  \anchor BaseObjectData_AdditionalNodes_header Functions to attach additional Nodes to objects
   *
   *  Manage additional scenegraph nodes that belong to an object.
   *
   * @{ */
  //===========================================================================
  public:
    /** \brief add an additional node to the object
    *
    * This function can be used to store an additional Scenegraph node. If you add nodes there, you do not
    * need to keep track of deleted, added objects, as this will be done in the main application.
    * You have to create the node yourself as this function does not know the type. You should add the
    * new node below the manipulatorNode ( BaseObjectData::manipulatorNode() ) if you want that it moves with the rest of the data. Otherwise
    * add it below the baseNode ( BaseObjectData::baseNode() of the object.
    *
    *  @param _node Node to add
    *  @param _pluginName Name of the current plugin
    *  @param _nodeName Name of the New Node
    *  @param _id extra index, if there will be multiple nodes with this name( defaults to 0)
    *  @return true if successfull
    */
    template< typename NodeT >
    bool addAdditionalNode(NodeT* _node , QString _pluginName, QString _nodeName , int _id = 0);

    /** \brief check if an object has the additional node
    *
    * If you store additional Scenegraph nodes with the objects you can check if they
    * exist with this function.
    *
    *  @param _pluginName Name of the current plugin
    *  @param _nodeName Name of the Node
    *  @param _id extra index, if there are multiple nodes with this name( defaults to 0)
    *  @return true if found
    */
    bool hasAdditionalNode(QString _pluginName, QString _nodeName , int _id = 0);

    /** \brief get an addition node from the object
    *
    * If you store additional Scenegraph node with the objects you can get these nodes with this function.
    *
    *  @param _node Returns the node
    *  @param _pluginName Name of the current plugin
    *  @param _nodeName Name of the Node
    *  @param _id extra index, if there are multiple nodes with this name ( defaults to 0)
    *  @return true if found
    */
    template< typename NodeT >
    bool getAdditionalNode(NodeT*& _node , QString _pluginName, QString _nodeName , int _id = 0 );

    /** \brief remove an additional node from the object
    *
    * If additional nodes are stored for this object, such a node can be removed using this function.
    * If this node has children, they will be removed from the scenegraph as well (and their
    * memory is freed).
    *
    *  @param _node Needed for type specification
    *  @param _pluginName Name of the current plugin
    *  @param _nodeName Name of the Node
    *  @param _id extra index, if there are multiple nodes with this name ( defaults to 0)
    *  @return true if found and removed
    */
    template< typename NodeT >
    bool removeAdditionalNode(NodeT*& _node, QString _pluginName, QString _nodeName , int _id = 0 );
  private:
    /** This pointer may be used to store additional Nodes belonging to this Object
     *   The String should be used by the plugin to identify its Nodes
     */
    std::vector< std::pair <BaseNode*,QString> > additionalNodes_;

  /** @} */

};


//=============================================================================

#if defined(INCLUDE_TEMPLATES) && !defined(BASEOBJECTDATA_C)
#define BASEOBJECT_TEMPLATES
#include "BaseObjectDataT_impl.hh"
#endif

