/*===========================================================================*\
 *                                                                           *
 *                             OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
 *                                                                           *
\*===========================================================================*/

#pragma once

#define TEXTUREINDEX "OriginalTexIndexMapping"

//=== INCLUDES ================================================================


// STL
#include <vector>

// OpenMesh
#include <OpenFlipper/common/GlobalDefines.hh>
#include <OpenMesh/Core/Geometry/VectorT.hh>
#include <ObjectTypes/PolyMesh/PolyMesh.hh>
#include <ObjectTypes/TriangleMesh/TriangleMesh.hh>
#include <OpenFlipper/common/BaseObject.hh>


#ifdef ENABLE_BSPLINECURVE_SUPPORT
#include <ObjectTypes/BSplineCurve/BSplineCurve.hh>
#endif

#ifdef ENABLE_BSPLINESURFACE_SUPPORT
#include <ObjectTypes/BSplineSurface/BSplineSurface.hh>
#endif

#include "Material.hh"

//=== IMPLEMENTATION ==========================================================

typedef int VertexHandle;
typedef int FaceHandle;
typedef std::vector<VertexHandle> VHandles;
typedef std::vector<OpenMesh::VertexHandle> OMVHandles;
typedef OpenMesh::Vec3f Vec3f;
typedef OpenMesh::Vec2f Vec2f;
typedef OpenMesh::Vec3uc Vec3uc;
typedef OpenMesh::Vec4uc Vec4uc;



class OBJImporter
{
  public:

    enum ObjectOptionsE
    {
      NONE             = 0,
      TRIMESH          = 1,
      POLYMESH         = 1 << 1,
      CURVE            = 1 << 2,
      SURFACE          = 1 << 3,
      NORMALS          = 1 << 4,
      TEXCOORDS        = 1 << 5,
      FACECOLOR        = 1 << 6,
      TEXTURE          = 1 << 7,
      FORCE_NOCOLOR    = 1 << 8,
      FORCE_NONORMALS  = 1 << 9,
      FORCE_NOTEXTURES = 1 << 10
    };
    
    typedef unsigned int ObjectOptions;

    /// Constructor
    OBJImporter() : degreeU_(0),degreeV_(0),currentGroup_(0) {
        // Add default group
        addGroup("DefaultGroup");
    }

    /// base class needs virtual destructor
    ~OBJImporter();

    /// add a vertex with coordinate \c _point
    VertexHandle addVertex(const Vec3f& _point);

    /// get vertex with given index
    Vec3f vertex(unsigned int _index);
    
    /// add texture coordinates
    int addTexCoord(const Vec2f& _coord);
    
    /// add a normal
    int addNormal(const Vec3f& _normal);
    
    /// set degree
    void setDegreeU(int _degree);
    void setDegreeV(int _degree);
    
    /// get current degree
    int degreeU();
    int degreeV();
    
    /// add an object
    void setObject( BaseObject* _object, int _groupId );
    
    /**\brief Get the id of the current group
     *
     * OBJ files can contain groups which are handled inside OpenFlipper as separate Objects.
     * When loading a file, the importer has to keep track of the current group and store a state
     * for each group with its properties (materials) )
     *
     * @return Id of the current group
     */
    int currentGroup();
    
    /// get a pointer to the active polyMesh
    PolyMesh* currentPolyMesh();
    
    /// get a pointer to the active triMesh
    TriMesh* currentTriMesh();

#ifdef ENABLE_BSPLINECURVE_SUPPORT
    BSplineCurve* currentCurve();
    /// returns the number of curves
    unsigned int numCurves() { return curvesMap_.size(); }
    /// maps the counted curve to a group id
    void setCurveGroupId(const unsigned int _count, const int _id);
    /// get the group id corresponding to the counted curve
    int getCurveGroupId(const unsigned int _count);
    /// sets the parent group id of the curve group
    void setCurveParentId(const int _curveGroup, const int _parentGroup);
    /// get the parent group id of the curve group
    int getCurveParentId(const int _curveGroup);
#endif

#ifdef ENABLE_BSPLINECURVE_SUPPORT
    BSplineSurface* currentSurface();
    /// returns the number of surfaces
    unsigned int numSurfaces() { return surfacesMap_.size(); }
    /// maps the counted surface to a group id
    void setSurfaceGroupId(const unsigned int _count, const int _id);
    /// get the group id corresponding to the counted surface
    int getSurfaceGroupId(const unsigned int _count);
    /// sets the parent group id of the surface group
    void setSurfaceParentId(const int _surfaceGroup, const int _parentGroup);
    /// get the parent group id of the surface group
    int getSurfaceParentId(const int _surfaceGroup);
#endif

    /// add all vertices that are used to the mesh (in correct order)
    void addUsedVertices(int _groupId);
    
    /// set vertex texture coordinate
    void setVertexTexCoord(VertexHandle _vh, int _texCoordID);
    
    /// set vertex normal
    void setNormal(int _index, int _normalID);

    /// add a face with indices _indices refering to vertices
    void addFace(const VHandles& _indices);
    
    /// add face and texture coordinates
    void addFace(const VHandles& _indices, const std::vector<int>& _face_texcoords);
    
  private:
    bool addFace(const VHandles& _indices, OpenMesh::FaceHandle &_outFH, std::vector< TriMesh::VertexHandle > &_outTriVertices, std::vector< PolyMesh::VertexHandle > &_outPolyVertices);
  public:

    /// force all meshes to be opened with specific type
    void forceMeshType( ObjectOptions _meshType );
    
    /// Query Object Options
    bool hasNormals(int _objectID);
    bool hasTexture(int _objectID);
    bool hasTextureCoords(int _objectID);
    bool isTriangleMesh(int _objectID);
    bool isPolyMesh(int _objectID);
    bool isCurve(int _objectID);
    bool isSurface(int _objectID);
    bool isNone(int _objectID);

    /// Set Object Option
    void setOption( ObjectOptionsE _option);

    /// Set Object Option
    void setOption( ObjectOptionsE _option, int _groupId);

    /// Global Properties
    unsigned int n_vertices();
    unsigned int n_normals();
    unsigned int n_texCoords();

    /** \brief Number of groups currently stored in the importer
     *
     * As OBJ files can contain several groups, we need to store information per group.
     *
     * @return Number of groups currently found by the importer
     */
    unsigned int groupCount();
    
    /// return object for the given group
    BaseObject* object(int _groupId );
    
    /// return all loaded materials
    MaterialList& materials();
    
    /// Add a material
    void addMaterial(std::string _materialName);
    
    /// used materials
    const std::vector<std::string> usedMaterials(unsigned int _objectID);
    void useMaterial( std::string _materialName );
    
    ///used vertices
    void useVertex(int _vertex_index);
    
    /// Path of the OBJ file
    QString path();
    void setPath(QString _path);
    
    /// store an initial options object for an object
    /// containing info about the meshType
    void setObjectOptions(ObjectOptions _options);
    
    /// Return true if the importer has no options stored
    bool noOptions();
    
    /// check if object with given id has given option
    bool hasOption( unsigned int _id, ObjectOptions _option );
    
    /// change the name of an object
    void setObjectName(int _objectID, QString _name);
    
    // Add a new group
    int addGroup(const QString& _groupName);
    int groupId(const QString& _groupName) const;
    unsigned int numGroups() const { return groupNames_.size(); }
    const QString groupName(const int _grpId) const;
    void setGroupName(const int _grp, const QString& _name);

    void setCurrentGroup(const int _current);
    int currentGroup() const;

    /// Finish up importing:
    /// Duplicate vertices of non-manifold faces and
    /// add new face as isolated one
    void finish();

  private:

    bool vertexListIsManifold(const std::vector<PolyMesh::VertexHandle>& _vertices) const;

    // general data
    std::vector< Vec3f > vertices_;
    std::vector< Vec3f > normals_;
    std::vector< Vec2f > texCoords_;
    
    //stores half edge normals of the current face
    std::map<TriMesh::VertexHandle,TriMesh::Normal> storedTriHENormals_;
    std::map<TriMesh::VertexHandle,PolyMesh::Normal> storedPolyHENormals_;

    int degreeU_;
    int degreeV_;
    
    MaterialList materials_;
    
    QString path_;
    
    std::vector<QString> groupNames_;
    int currentGroup_;

    // polyMesh data
    std::vector<std::map< int, PolyMesh::VertexHandle > > vertexMapPoly_;
    
    PolyMesh::FaceHandle addedFacePoly_;
    
    // triMesh data
    std::vector<std::map< int, TriMesh::VertexHandle > > vertexMapTri_;
    
    std::vector<std::vector< TriMesh::FaceHandle > > addedFacesTri_;
    
    //std::vector< BaseObject* > objects_;

    //object data
    std::vector<TriMeshObject*> triMeshes_;
    std::vector<PolyMeshObject*> polyMeshes_;
#ifdef ENABLE_BSPLINECURVE_SUPPORT
    std::vector<BSplineCurveObject*> bSplineCurves_;
#endif
#ifdef ENABLE_BSPLINESURFACE_SUPPORT
    std::vector<BSplineSurfaceObject*> bSplineSurfaces_;
#endif

    std::vector< ObjectOptions > objectOptions_;
    
    // for each object a vector of materialNames
    std::vector< std::vector< std::string > > usedMaterials_;
    
    // for each object a vector of vertex indices
    // this ensures that a vertex defined first gets a lower index
    std::vector< std::map< int, VertexHandle > > usedVertices_;

    // Store vertices of invalid faces due to non-manifold
    // configurations.
    std::vector< std::vector< OMVHandles > > invalidFaces_;

#ifdef ENABLE_BSPLINECURVE_SUPPORT
    /// maps each counted curve to a group id
    std::map<unsigned int, int> curvesMap_;
    /// maps each curve group to a parent group id
    std::map<int, int> curveParentGroupMap_;
#endif

#ifdef ENABLE_BSPLINESURFACE_SUPPORT
    /// maps each counted curve to a group id
    std::map<unsigned int, int> surfacesMap_;
    /// maps each surface group  to a parent group id
    std::map<int, int> surfaceParentGroupMap_;
#endif

};

