/*===========================================================================*\
*                                                                            *
*                              OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
*                                                                            *
\*===========================================================================*/




#define OM_PROPERTY_VISUALIZER_CC

#include "OMPropertyVisualizer.hh"



namespace {

template <typename EntityType, typename Handle, typename MeshT>
QString getPropertyText__(Handle handle, MeshT mesh, const PropertyInfo& propInfo)
{
    EntityType prop;

    if ( !mesh->get_property_handle(prop, propInfo.propName() ) )
        return QObject::tr("Error: No property with name ").append(propInfo.propName().c_str());

    return PropertyVisualizer::toStr(mesh->property(prop, handle));
}

}


template <typename MeshT>
template <typename InnerType>
QString OMPropertyVisualizer<MeshT>::getPropertyText_(unsigned int index)
{
    MeshT* mesh = OMPropertyVisualizer<MeshT>::mesh;
    if (PropertyVisualizer::propertyInfo.isFaceProp())
        return getPropertyText__<OpenMesh::FPropHandleT< InnerType > >(mesh->face_handle(index),OMPropertyVisualizer<MeshT>::mesh, PropertyVisualizer::propertyInfo);
    else if (PropertyVisualizer::propertyInfo.isEdgeProp())
        return getPropertyText__<OpenMesh::EPropHandleT< InnerType > >(mesh->edge_handle(index),OMPropertyVisualizer<MeshT>::mesh, PropertyVisualizer::propertyInfo);
    else if (PropertyVisualizer::propertyInfo.isHalfedgeProp())
        return getPropertyText__<OpenMesh::HPropHandleT< InnerType > >(mesh->halfedge_handle(index),OMPropertyVisualizer<MeshT>::mesh, PropertyVisualizer::propertyInfo);
    else //if (propertyInfo.isVertexProp())
        return getPropertyText__<OpenMesh::VPropHandleT< InnerType > >(mesh->vertex_handle(index),OMPropertyVisualizer<MeshT>::mesh, PropertyVisualizer::propertyInfo);
}


template<typename MeshT>
template<typename PropHandle>
void OMPropertyVisualizer<MeshT>::removeProperty_stage2() {
    PropHandle propHandle;
    if (!mesh->get_property_handle(propHandle, propertyInfo.propName())) return;
    mesh->remove_property(propHandle);
}

template<typename MeshT>
template<typename PropType>
inline void OMPropertyVisualizer<MeshT>::removeProperty_stage1() {
    if (propertyInfo.isEdgeProp())
        removeProperty_stage2<OpenMesh::EPropHandleT<PropType> >();
    else if (propertyInfo.isVertexProp())
        removeProperty_stage2<OpenMesh::VPropHandleT<PropType> >();
    else if (propertyInfo.isFaceProp())
        removeProperty_stage2<OpenMesh::FPropHandleT<PropType> >();
    else if (propertyInfo.isHalfedgeProp())
        removeProperty_stage2<OpenMesh::HPropHandleT<PropType> >();
}


template<typename MeshT>
template<typename PropHandle, typename Iterator>
inline void OMPropertyVisualizer<MeshT>::duplicateProperty_stage2(Iterator first, Iterator last) {
    PropHandle propHandle;
    if (!mesh->get_property_handle(propHandle, propertyInfo.propName())) return;

    std::string newPropertyName;
    for (int i = 1;; ++i) {
        std::ostringstream oss;
        oss << propertyInfo.propName() << " Copy " << i;
        newPropertyName = oss.str();

        PropHandle tmp;
        if (!mesh->get_property_handle(tmp, newPropertyName)) break;
    }

    PropHandle newProperty;
    mesh->add_property(newProperty, newPropertyName);
    std::for_each(first, last, CopyProperty<PropHandle>(newProperty, propHandle, mesh));
}

template<typename MeshT>
template<typename PropType>
inline void OMPropertyVisualizer<MeshT>::duplicateProperty_stage1() {
    if (propertyInfo.isEdgeProp())
        duplicateProperty_stage2<OpenMesh::EPropHandleT<PropType> >(mesh->edges_begin(), mesh->edges_end());
    else if (propertyInfo.isVertexProp())
        duplicateProperty_stage2<OpenMesh::VPropHandleT<PropType> >( mesh->vertices_begin(), mesh->vertices_end());
    else if (propertyInfo.isFaceProp())
        duplicateProperty_stage2<OpenMesh::FPropHandleT<PropType> >(mesh->faces_begin(), mesh->faces_end());
    else if (propertyInfo.isHalfedgeProp())
        duplicateProperty_stage2<OpenMesh::HPropHandleT<PropType> >(mesh->halfedges_begin(), mesh->halfedges_end());
}



template <typename MeshT>
QString OMPropertyVisualizer<MeshT>::getPropertyText(unsigned int index)
{
    return "";
}


template <typename MeshT>
void OMPropertyVisualizer<MeshT>::setDrawMode(const ACG::SceneGraph::DrawModes::DrawMode& _mode)
{
  BaseObjectData* object;
  if (PluginFunctions::getObject(mObjectID, object))
    object->setObjectDrawMode(_mode);
}



template <typename MeshT>
void OMPropertyVisualizer<MeshT>::setPropertyFromText(unsigned int index, QString text)
{
    if (propertyInfo.isFaceProp())
        setFacePropertyFromText(index, text);
    else if (propertyInfo.isEdgeProp())
        setEdgePropertyFromText(index, text);
    else if (propertyInfo.isHalfedgeProp())
        setHalfedgePropertyFromText(index, text);
    else //if (propertyInfo.isVertexProp())
        setVertexPropertyFromText(index, text);
}

template <typename MeshT>
int OMPropertyVisualizer<MeshT>::getEntityCount()
{
    if (propertyInfo.isFaceProp())
        return mesh->n_faces();
    else if (propertyInfo.isEdgeProp())
        return mesh->n_edges();
    else if (propertyInfo.isHalfedgeProp())
        return mesh->n_halfedges();
    else //if (propertyInfo.isVertexProp())
        return mesh->n_vertices();
}

template <typename MeshT>
QString OMPropertyVisualizer<MeshT>::getHeader()
{
    //Header: headerVersion, numberOfEntities, typeOfEntities, typeOfProperty, propertyName

    QString header = QObject::tr("1"); //version
    header.append(QObject::tr(", %1").arg(getEntityCount())); //number of entities
    header.append(QObject::tr(", %1").arg(propertyInfo.entityType())); //type of entities
    header.append(", ").append(propertyInfo.friendlyTypeName()); //type of property
    header.append(", ").append(propertyInfo.propName().c_str()); // name of property
    return header;
}


/**
 * @brief Returns the ID of the closest primitive.
 *
 * @param _face The face that was hit.
 * @param _hitPoint The position where the face was hit.
 * @return The ID of the primitive that is closest to the hitpoint.
 *
 * This method is used for picking. For some entities it might be convenient to not pick them
 * directly but pick faces instead. Then, this method iterates over the entities adjacent to the face
 * and return the one that is closest to the actual hit point.
 */
template <typename MeshT>
unsigned int OMPropertyVisualizer<MeshT>::getClosestPrimitiveId(unsigned int _face, ACG::Vec3d& _hitPoint)
{
    if (propertyInfo.isFaceProp())
        return getClosestFaceId(_face, _hitPoint);
    else if (propertyInfo.isEdgeProp())
        return getClosestEdgeId(_face, _hitPoint);
    else if (propertyInfo.isHalfedgeProp())
        return getClosestHalfedgeId(_face, _hitPoint);
    else //if (propertyInfo.isVertexProp())
        return getClosestVertexId(_face, _hitPoint);
}

template <typename MeshT>
unsigned int OMPropertyVisualizer<MeshT>::getClosestFaceId(unsigned int _face, ACG::Vec3d& _hitPoint)
{
    return _face;
}

template <typename MeshT>
unsigned int OMPropertyVisualizer<MeshT>::getClosestEdgeId(unsigned int _face, ACG::Vec3d& _hitPoint)
{
    unsigned int closest_halfedge_id = getClosestHalfedgeId(_face, _hitPoint);
    typename MeshT::HalfedgeHandle heh;
    heh = mesh->halfedge_handle(closest_halfedge_id);

    typename MeshT::EdgeHandle eh;
    eh = mesh->edge_handle(heh);

    return eh.idx();
}

template <typename MeshT>
unsigned int OMPropertyVisualizer<MeshT>::getClosestHalfedgeId(unsigned int _face, ACG::Vec3d& _hitPoint)
{
    typename MeshT::FaceHalfedgeIter fh_it;

    int closest_h_idx = 0;
    double dist = DBL_MAX;

    ACG::Vec3d vTemp = ACG::Vec3d(0.0, 0.0, 0.0);
    typename MeshT::Point p;

    for (fh_it = mesh->fh_iter(mesh->face_handle(_face)); fh_it.is_valid(); ++fh_it){

        typename MeshT::HalfedgeHandle heh;
        heh = *fh_it;

        typename MeshT::VertexHandle v1;
        v1 = mesh->to_vertex_handle(heh);
        typename MeshT::VertexHandle v2;
        v2 = mesh->from_vertex_handle(heh);

        p = 0.5* (mesh->point( v1 ) + mesh->point( v2 ));

        // Find closest vertex to selection
        vTemp = ACG::Vec3d(p[0], p[1], p[2]);
        const double temp_dist = (vTemp - _hitPoint).length();

        if (temp_dist < dist) {
            dist = temp_dist;
            closest_h_idx = fh_it->idx();
        }

    }
    return closest_h_idx;
}

template <typename MeshT>
unsigned int OMPropertyVisualizer<MeshT>::getClosestVertexId(unsigned int _face, ACG::Vec3d& _hitPoint)
{
    typename MeshT::FaceVertexIter fv_it;

    int closest_v_idx = 0;
    double dist = DBL_MAX;

    ACG::Vec3d vTemp = ACG::Vec3d(0.0, 0.0, 0.0);
    typename MeshT::Point p;

    for (fv_it = mesh->fv_iter(mesh->face_handle(_face)); fv_it.is_valid(); ++fv_it){

      p = mesh->point( *fv_it );

      // Find closest vertex to selection
      vTemp = ACG::Vec3d(p[0], p[1], p[2]);
      const double temp_dist = (vTemp - _hitPoint).length();

      if (temp_dist < dist) {
          dist = temp_dist;
          closest_v_idx = fv_it->idx();
      }

    }
    return closest_v_idx;
}

/**
 * \brief Visualizes the property.
 *
 * To visualize the property this method calls the corresponding visualization method
 * visualizeFaceProp, -EdgeProp, -HalfedgeProp or -VertexProp depending on the entity
 * type of the property.
 */
template <typename MeshT>
void OMPropertyVisualizer<MeshT>::visualize(bool _setDrawMode, QWidget* _widget)
{
    QWidget* tmp;
    if (_widget)
    {
        tmp = widget;
        widget = _widget;
    }

    if (propertyInfo.isFaceProp())
        visualizeFaceProp(_setDrawMode);
    else if (propertyInfo.isEdgeProp())
        visualizeEdgeProp(_setDrawMode);
    else if (propertyInfo.isHalfedgeProp())
        visualizeHalfedgeProp(_setDrawMode);
    else if (propertyInfo.isVertexProp())
        visualizeVertexProp(_setDrawMode);

    if (_widget)
    {
        widget = tmp;
    }
}

/**
 * @brief Clears the property.
 *
 * This method clears the property by releasing the colors of the mesh.
 */
template <typename MeshT>
void OMPropertyVisualizer<MeshT>::clear()
{
    if (propertyInfo.isFaceProp())
        clearFaceProp();
    else if (propertyInfo.isEdgeProp())
        clearEdgeProp();
    else if (propertyInfo.isHalfedgeProp())
        clearHalfedgeProp();
    else if (propertyInfo.isVertexProp())
        clearVertexProp();
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::visualizeFaceProp(bool /*_setDrawMode*/)
{
    emit log(LOGERR, "Visualizing FaceProp not implemented");
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::visualizeEdgeProp(bool /*_setDrawMode*/)
{
    emit log(LOGERR, "Visualizing EdgeProp not implemented");
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::visualizeHalfedgeProp(bool /*_setDrawMode*/)
{
    emit log(LOGERR, "Visualizing HalfedgeProp not implemented");
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::visualizeVertexProp(bool /*_setDrawMode*/)
{
    emit log(LOGERR, "Visualizing VertexProp not implemented");
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::clearFaceProp()
{
    if ( mesh->has_face_colors() )
        OMPropertyVisualizer<MeshT>::mesh->release_face_colors();
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::clearEdgeProp()
{
    if ( mesh->has_edge_colors() )
        OMPropertyVisualizer<MeshT>::mesh->release_edge_colors();
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::clearHalfedgeProp()
{
    if ( mesh->has_halfedge_colors() )
        OMPropertyVisualizer<MeshT>::mesh->release_halfedge_colors();
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::clearVertexProp()
{
    if ( mesh->has_vertex_colors() )
        OMPropertyVisualizer<MeshT>::mesh->release_vertex_colors();
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::setFacePropertyFromText(unsigned int index, QString text)
{
    emit log(LOGERR, "Setting face property not implemented");
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::setEdgePropertyFromText(unsigned int index, QString text)
{
    emit log(LOGERR, "Settingedge property not implemented");
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::setHalfedgePropertyFromText(unsigned int index, QString text)
{
    emit log(LOGERR, "Setting halfedge property not implemented");
}

template <typename MeshT>
void OMPropertyVisualizer<MeshT>::setVertexPropertyFromText(unsigned int index, QString text)
{
    emit log(LOGERR, "Setting vertex property not implemented");
}


template<typename MeshT>
template<typename Type>
void OMPropertyVisualizer<MeshT>::showHistogram(ACG::QtWidgets::QtHistogramWidget *histogramWidget) {
    using PV = OMPropertyVisualizer<MeshT>;
    const std::string &prop_name = PV::propertyInfo.propName();

    // ugly repetition ahead. In C++14, we could use a generic lambda,
    // in C++11 the cleanest solution would be to add another templated member function - not worth it.
    switch (PropertyVisualizer::propertyInfo.entityType()) {
    case PropertyInfo::EF_FACE:
    {
        OpenMesh::FPropHandleT<Type> prop_handle;
        if (!PV::mesh->get_property_handle(prop_handle, prop_name)) break;
        PropertyVisualizer::template showHistogramT<Type>(
                    histogramWidget,
                    PV::mesh->property(prop_handle).data_vector());
        break;
    }
    case PropertyInfo::EF_EDGE:
    {
        OpenMesh::EPropHandleT<Type> prop_handle;
        if (!PV::mesh->get_property_handle(prop_handle, prop_name)) break;
        PropertyVisualizer::template showHistogramT<Type>(
                    histogramWidget,
                    PV::mesh->property(prop_handle).data_vector());
        break;
    }
    case PropertyInfo::EF_HALFEDGE:
    {
        OpenMesh::HPropHandleT<Type> prop_handle;
        if (!PV::mesh->get_property_handle(prop_handle, prop_name)) break;
        PropertyVisualizer::template showHistogramT<Type>(
                    histogramWidget,
                    PV::mesh->property(prop_handle).data_vector());
        break;
    }
    case PropertyInfo::EF_VERTEX:
    {
        OpenMesh::VPropHandleT<Type> prop_handle;
        if (!PV::mesh->get_property_handle(prop_handle, prop_name)) break;
        PropertyVisualizer::template showHistogramT<Type>(
                    histogramWidget,
                    PV::mesh->property(prop_handle).data_vector());
        break;
    }
    default:
        assert(false);
    }
}
