/*===========================================================================*\
*                                                                            *
*                              OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
*                                                                            *
\*===========================================================================*/






//=============================================================================
//
//  CLASS Core - IMPLEMENTATION of Comunication with plugins
//
//=============================================================================


//== INCLUDES =================================================================

#include "Core.hh"

#include "OpenFlipper/BasePlugin/PluginFunctionsCore.hh"
#include <stdexcept>

//== IMPLEMENTATION ==========================================================

//========================================================================================
// ===             Object List Communication                       =======================
//========================================================================================

/** \brief Handle object updates by plugins
 *
 * This function is called by a plugin if it changed something in the object list (source,target, or other properties ).
 * The information is passed to all plugins.
 *
 * @param _identifier Id of the updated object
 * @param _type       What part of the object has been updated
 */
void Core::slotObjectUpdated(int _identifier, const UpdateType& _type) {
  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"updatedObject( " + QString::number(_identifier) + ", " + updateTypeName(_type)
                 + tr(" ) called by ") + QString( sender()->metaObject()->className() ) );
      }
    } else {
      emit log(LOGINFO,"updatedObject( " + QString::number(_identifier) + ", " + updateTypeName(_type) + tr(" ) called by Core") );
    }
  }

  // Disable redraws as everything here has to update the object only once
  OpenFlipper::Options::redrawDisabled(true);

  // If we are called for a special object, we update it ourself so the Plugins dont need to do that.
  BaseObject* object = 0;
  if ( _identifier != -1 ) {
    if ( !PluginFunctions::getObject(_identifier,object) ) {
      emit log(LOGERR,tr("updated_objects called for non existing object with id : ") + QString::number(_identifier) );
      return;
    }
  }


  // If the identifier is -1 we force an update of all objects in the scene (Bad idea for scenes with many objects)
  if ( _identifier == -1 ) {

    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGWARN,"updatedObject( " + QString::number(_identifier) + ", " + updateTypeName(_type)
            + tr(" ) called by ") + QString( sender()->metaObject()->className() + tr(" which is inefficient)") ) );
      }
    } else {
      emit log(LOGWARN,"updatedObject( " + QString::number(_identifier) + ", " + updateTypeName(_type) + tr(" ) called by Core, which is inefficient!") );
    }

    for ( PluginFunctions::ObjectIterator o_it(PluginFunctions::ALL_OBJECTS,DATA_ALL ) ;  o_it != PluginFunctions::objectsEnd(); ++o_it) {

      // just inform the plugins as we don't do anything else (Do deprecated and new updatedObjects here)
      emit signalObjectUpdated(o_it->id());
      emit signalObjectUpdated(o_it->id(), _type);

      // Call the objects update function
      o_it->update();

    }
  } else {
    // just inform the plugins as we don't do anything else (Do deprecated and new updatedObjects here)
    emit signalObjectUpdated(_identifier);
    emit signalObjectUpdated(_identifier, _type);
  }


  // If we have an single object, call it's update function
  if ( object != 0 )
    object->update(_type);

  // Reenable redraws
  OpenFlipper::Options::redrawDisabled(false);


  // Reset scenegraph but keep scene center!
  resetScenegraph(false);

  updateView();
}

void Core::slotVisibilityChanged( int _id ) {

  // tell plugins
  emit visibilityChanged( _id );

  // Reset scenegraph but keep scene center!
  resetScenegraph(false);
  
  updateView();
}

/** This function is called if the active object has changed. The information is passed to all plugins.
*/
void Core::slotObjectSelectionChanged( int _id )
{
  // just inform the plugins as we don't do anything else
  emit objectSelectionChanged(_id);

  // Send via second interface
  emit signalObjectUpdated(_id,UPDATE_STATE);

  updateView();
}

void Core::slotObjectPropertiesChanged( int _id )
{
  emit objectPropertiesChanged(_id);
}

//========================================================================================
// ===             Texture Communication                       ===========================
//========================================================================================


/** Called by a plugin if it created a texture. The information is passed to all plugins. If a texture control plugin is available it has to react on the signal.\n
 * See in the documentation of the texture plugin interfaces for further detail.
*/
void Core::slotAddTexture( QString _textureName , QString _filename, uint _dimension, int _id) {

  if (QThread::currentThread() != QApplication::instance()->thread())
  {
    //execute method in main thread
    QMetaObject::invokeMethod(this,"slotAddTexture",Qt::BlockingQueuedConnection, Q_ARG(QString, _textureName), Q_ARG(QString, _filename), Q_ARG(uint, _dimension), Q_ARG(int, _id));
    return;
  }

  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"addTexture( " + _textureName + "," + _filename + ","  +  QString::number(_dimension) + ", " + QString::number(_id) +  tr(" ) called by ") +
        QString( sender()->metaObject()->className() ) );
      }
    } else {
      emit log(LOGINFO,"addTexture( " + _textureName + "," + _filename + "," +  QString::number(_dimension) + ", " + QString::number(_id) +  tr(" ) called by Core") );
    }
  }
  
  emit addTexture(_textureName , _filename,_dimension,_id);
}

/** Called by a plugin if it created a texture. The information is passed to all plugins. If a texture control plugin is available it has to react on the signal.\n
 * See in the documentation of the texture plugin interfaces for further detail.
*/
void Core::slotAddTexture( QString _textureName , QImage _image, uint _dimension, int _id) {

  if (QThread::currentThread() != QApplication::instance()->thread())
  {
    //execute method in main thread
    QMetaObject::invokeMethod(this,"slotAddTexture",Qt::BlockingQueuedConnection, Q_ARG(QString, _textureName), Q_ARG(QImage, _image), Q_ARG(uint, _dimension), Q_ARG(int, _id));
    return;
  }

  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"addTexture( " + _textureName + ",QImage,"  +  QString::number(_dimension) + ", " + QString::number(_id) +  tr(" ) called by ") +
        QString( sender()->metaObject()->className() ) );
      }
    } else {
      emit log(LOGINFO,"addTexture( " + _textureName + ",image," +  QString::number(_dimension) + ", " + QString::number(_id) +  tr(" ) called by Core") );
    }
  }
  
  emit addTexture(_textureName , _image,_dimension,_id);
}

/** Called by a plugin if it created a texture. The information is passed to all plugins. If a texture control plugin is available it has to react on the signal.\n
 * See in the documentation of the texture plugin interfaces for further detail.
*/
void Core::slotAddTexture( QString _textureName , QString _filename, uint _dimension) {
  
  if (QThread::currentThread() != QApplication::instance()->thread())
  {
    //execute method in main thread
    QMetaObject::invokeMethod(this,"slotAddTexture",Qt::BlockingQueuedConnection, Q_ARG(QString, _textureName), Q_ARG(QString, _filename), Q_ARG(uint, _dimension));
    return;
  }

  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"slotAddTexture( " + _textureName + "," + _filename + ","  +  QString::number(_dimension)  +  tr(" ) called by ") +
        QString( sender()->metaObject()->className() ) );
      }
    } else {
      emit log(LOGINFO,"slotAddTexture( " + _textureName + "," + _filename + "," +", " +  QString::number(_dimension) + tr(" ) called by Core") );
    }
  }
  
  emit addTexture(_textureName , _filename,_dimension);
}

/** Called by a plugin if it created a texture. The information is passed to all plugins. If a texture control plugin is available it has to react on the signal.\n
 * See in the documentation of the texture plugin interfaces for further detail.
*/
void Core::slotAddTexture( QString _textureName , QImage _image, uint _dimension) {
  
  if (QThread::currentThread() != QApplication::instance()->thread())
  {
    //execute method in main thread
    QMetaObject::invokeMethod(this,"slotAddTexture",Qt::BlockingQueuedConnection, Q_ARG(QString, _textureName), Q_ARG(QImage, _image), Q_ARG(uint, _dimension));
    return;
  }

  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"slotAddTexture( " + _textureName + ",_image,"  +  QString::number(_dimension)  +  tr(" ) called by ") +
        QString( sender()->metaObject()->className() ) );
      }
    } else {
      emit log(LOGINFO,"slotAddTexture( " + _textureName + ",_image," +", " +  QString::number(_dimension) + tr(" ) called by Core") );
    }
  }
  
  emit addTexture(_textureName , _image,_dimension);
}

/** Called by a plugin if a texture has to be updated. The information is passed to all plugins. The Plugin providing the given Texture should react on this event.\n
 * See in the documentation of the texture plugin interfaces for further detail.
*/
void Core::slotUpdateTexture( QString _name , int _identifier){

  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"slotUpdateTexture( "  + _name + " , " + QString::number(_identifier) + tr(" ) called by ") +
                 QString( sender()->metaObject()->className() ) );
      }
    }
  }

  emit updateTexture(_name, _identifier);
}

void Core::slotMultiTextureAdded( QString _textureGroup , QString _name , QString _filename , int _id , int* _textureId ) {
  slotMultiTextureAdded(_textureGroup, _name, _filename, _id, *_textureId);
}

void Core::slotMultiTextureAdded( QString _textureGroup , QString _name , QString _filename , int _id , int& _textureId ) {
  
  if (QThread::currentThread() != QApplication::instance()->thread())
  {
    //execute method in main thread
    QMetaObject::invokeMethod(this,"slotMultiTextureAdded",Qt::BlockingQueuedConnection, Q_ARG(QString, _textureGroup), Q_ARG(QString, _name), Q_ARG(QString, _filename), Q_ARG(int, _id), Q_ARG(int*, &_textureId));
    return;
  }

  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"slotMultiTextureAdded( " + _textureGroup + ", " + _name + "," + _filename + ","  + QString::number(_id) + tr(" ) called by ") +
        QString( sender()->metaObject()->className() ) );
      }
    } else {
      emit log(LOGINFO,"slotMultiTextureAdded( " + _textureGroup + ", " + _name + "," + _filename + ","  + QString::number(_id) + tr(" ) called by Core") );
    }
  }
  
  emit addMultiTexture( _textureGroup , _name , _filename , _id , _textureId  );
}

void Core::slotMultiTextureAdded( QString _textureGroup , QString _name , QImage _image , int _id , int& _textureId ) {
  
  if (QThread::currentThread() != QApplication::instance()->thread())
  {
    //execute method in main thread
    QMetaObject::invokeMethod(this,"slotMultiTextureAdded",Qt::BlockingQueuedConnection, Q_ARG(QString, _textureGroup), Q_ARG(QString, _name), Q_ARG(QImage, _image), Q_ARG(int, _id), Q_ARG(int*, &_textureId));
    return;
  }

  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"slotMultiTextureAdded( " + _textureGroup + ", " + _name + ",_image,"  + QString::number(_id) + tr(" ) called by ") +
        QString( sender()->metaObject()->className() ) );
      }
    } else {
      emit log(LOGINFO,"slotMultiTextureAdded( " + _textureGroup + ", " + _name + ",_image,"  + QString::number(_id) + tr(" ) called by Core") );
    }
  }
  
  emit addMultiTexture( _textureGroup , _name , _image , _id , _textureId  );
}

/** Called by a plugin if all textures should be updated. The information is passed to all plugins. All plugins providing textures should react on this event.\n
 * See in the documentation of the texture plugin interfaces for further detail.
*/
void Core::slotUpdateAllTextures( ){
  emit updateAllTextures();
}

/** Called by a plugin if the parameters of a texture should be changed. The information is passed to all plugins. A Texturecontrol plugin should react on this event.\n
 * See in the documentation of the texture plugin interfaces for further detail.
*/
void Core::slotSetTextureMode(QString _textureName, QString _mode, int _id) {

  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"slotSetTextureMode( " + _textureName + " , " + _mode + " , " + QString::number(_id) + tr(" ) called by ") +
                 QString( sender()->metaObject()->className() ) );
      }
    }
  }

  emit setTextureMode(_textureName,_mode,_id);
}

/** Called by a plugin if the parameters of a texture should be changed. The information is passed to all plugins. A Texturecontrol plugin should react on this event.\n
 * See in the documentation of the texture plugin interfaces for further detail.
*/
void Core::slotSetTextureMode(QString _textureName ,QString _mode) {

  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"slotSetTextureMode( " + _textureName + " , " + _mode + tr(" ) called by ") +
                 QString( sender()->metaObject()->className() ) );
      }
    }
  }

  emit setTextureMode(_textureName,_mode);
}

/** Called by a plugin if it updated a texture. The information is passed to all plugins. If a texture control plugin is available it has to react on the signal and update the visualization of the texture.\n
 * See in the documentation of the texture plugin interfaces for further detail.
*/
void Core::slotTextureUpdated( QString _textureName , int _identifier ) {
  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"slotTextureUpdated( " + _textureName + " , " + QString::number(_identifier) + tr(" ) called by ") +
                 QString( sender()->metaObject()->className() ) );
      }
    }
  }
  emit updatedTextures(_textureName,_identifier);
}

/** Called by plugins if texture mode should be switched
 */
void Core::slotSwitchTexture( QString _textureName, int _id ) {
  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"switchTexture( " + _textureName + " , " + QString::number(_id) + tr(" ) called by ") +
            QString( sender()->metaObject()->className() ) );
      }
    }
  }
  emit switchTexture(_textureName, _id);
}

/** Called by plugins if texture mode should be switched
 */
void Core::slotSwitchTexture( QString _textureName ) {
  if ( OpenFlipper::Options::doSlotDebugging() ) {
    if ( sender() != 0 ) {
      if ( sender()->metaObject() != 0 ) {
        emit log(LOGINFO,"switchTexture( " + _textureName + tr(" ) called by ") +
            QString( sender()->metaObject()->className() ) );
      }
    }
  }
  emit switchTexture(_textureName);
}


/** Called by plugins if texture image should be changed
 */
void Core::slotTextureChangeImage( QString _textureName , QImage& _image ) {
  emit textureChangeImage( _textureName ,_image );
}

/** Called by plugins if texture image should be changed
 */
void Core::slotTextureChangeImage( QString _textureName , QImage& _image , int _id )  {
  emit textureChangeImage( _textureName , _image , _id );
}

/** Called by plugins if texture image should be fetched
 */
void Core::slotTextureGetImage( QString _textureName , QImage& _image ) {
  emit textureGetImage( _textureName ,_image );
}

/** Called by plugins if texture image should be fetched
 */
void Core::slotTextureGetImage( QString _textureName , QImage& _image , int _id )  {
  emit textureGetImage( _textureName , _image , _id );
}

/** Called by plugins if texture index should be fetched
 */
void Core::slotTextureIndex( QString _textureName, int _id, int& _index){
  emit textureIndex( _textureName, _id, _index);
}

/** Called by plugins if texture index property name should be fetched
 */
void Core::slotTextureIndexPropertyName( int _id, QString& _propertyName){
  emit textureIndexPropertyName( _id, _propertyName);
}

/** Called by plugins if texture name should be fetched
 */
void Core::slotTextureName( int _id, int _textureIndex, QString& _textureName){
  emit textureName( _id, _textureIndex, _textureName);
}

/** Called by plugins if texture filename should be fetched
 */
void Core::slotTextureFilename( int _id, QString _textureName, QString& _textureFilename){
  emit textureFilename( _id, _textureName, _textureFilename);
}

/** Called by plugins if current texture name should be returned
 */
void Core::slotGetCurrentTexture( int _id, QString& _textureName ){
  emit getCurrentTexture( _id, _textureName );
}

/** Called by plugins if texture image should be fetched
 */
void Core::slotGetSubTextures( int _id, QString _multiTextureName, QStringList& _subTextures ){
  emit getSubTextures( _id, _multiTextureName, _subTextures );
}

//========================================================================================
// ===            Object Manager                              ============================
//========================================================================================

/// This slot is called by the object manager when a new object is created
void Core::newObject(int _objectId) {
  
   BaseObject* baseObject = 0;
   PluginFunctions::getObject(_objectId,baseObject);
   
   if ( baseObject ) {
    connect( baseObject, SIGNAL(visibilityChanged(int)),       this, SLOT(slotVisibilityChanged(int)),       Qt::DirectConnection) ;
    connect( baseObject, SIGNAL(objectSelectionChanged(int)),  this, SLOT(slotObjectSelectionChanged(int)),  Qt::DirectConnection );
    connect( baseObject, SIGNAL(objectPropertiesChanged(int)), this, SLOT(slotObjectPropertiesChanged(int)), Qt::DirectConnection );
   } else {
     emit log(LOGERR,tr("newObject received from objectManager with invalid id! This should not happen. The new Object will not work correctly!"));
   }
   
}

/// This slot is called by the object manager when a object is deleted
void Core::deletedObject(int /*_objectId*/) {
}


//========================================================================================
// ===            Cross Plugin connections                    ============================
//========================================================================================

/// Helper function to avoid code duplication
void connectPlugins( Core* c, const std::vector<PluginInfo>& plugins_, QString _pluginName1, const char* _signal, QString _pluginName2, const char* _slot, bool _queued)
{
  QObject* plugin1 = 0;
  QObject* plugin2 = 0;

  for ( int i = 0 ; i < (int)plugins_.size(); ++i ) {
    if ( plugins_[i].rpcName == _pluginName1 ) {
      plugin1 = plugins_[i].plugin;
    }
    if ( plugins_[i].rpcName == _pluginName2 ) {
      plugin2 = plugins_[i].plugin;
    }
  }

  if (  plugin1 == 0 ) {
    emit c->log(LOGERR,QObject::tr("Cross Plugin Interconnection failed because plugin %1 was not found!").arg(_pluginName1));
    return;
  }

  if (  plugin2 == 0 ) {
    emit c->log(LOGERR,QObject::tr("Cross Plugin Interconnection failed because plugin %1 was not found!").arg(_pluginName2));
    return;
  }

  // now connect them
  if(_queued)
    QObject::connect(plugin1,_signal,plugin2,_slot, Qt::QueuedConnection);
  else
    QObject::connect(plugin1,_signal,plugin2,_slot);
}

void Core::slotCrossPluginConnect( QString _pluginName1, const char* _signal, QString _pluginName2, const char* _slot) {
  connectPlugins(this, plugins(), _pluginName1,  _signal, _pluginName2, _slot, false);
}

void Core::slotCrossPluginConnectQueued( QString _pluginName1, const char* _signal, QString _pluginName2, const char* _slot) {
  connectPlugins(this, plugins(), _pluginName1,  _signal, _pluginName2, _slot, true);
}

//========================================================================================
// ===            Renderer Settings                    ============================
//========================================================================================
void Core::slotSetRenderer(unsigned int _viewer, QString _rendererName) {
  renderManager().setActive(_rendererName,_viewer);
}

void Core::slotGetCurrentRenderer(unsigned int _viewer, QString& _rendererName) {
  _rendererName = renderManager().active(_viewer)->name;
}

void Core::slotMetadataDeserialized(
        const QVector<QPair<QString, QString> > &data) {

    QString obj_metadata;
    for (QVector<QPair<QString, QString> >::const_iterator
            it = data.begin(); it != data.end(); ++it) {
        if (it->first == "Mesh Comments")
            obj_metadata = it->second;

        emit genericMetadataDeserialized(it->first, it->second);
    }

    /*
     * Now handle object metadata.
     */
    QRegExp re_begin("BEGIN Comments for object \"([^\\n]*)\"");
    QRegExp re_end("\\nEND Comments for object \"([^\\n]*)\"");
    enum STATE {
        STATE_SEARCH_BEGIN,
        STATE_CONSUME_INNER,
        STATE_EOS,
    };

    int cursor = 0;
    QString current_object_name;
    for (STATE state = STATE_SEARCH_BEGIN; state != STATE_EOS; ) {
        switch (state) {
            case STATE_SEARCH_BEGIN:
                cursor = re_begin.indexIn(obj_metadata, cursor);
                if (cursor == -1) {
                    state = STATE_EOS;
                    break;
                }
                current_object_name = re_begin.cap(1);
                cursor += re_begin.matchedLength();
                state = STATE_CONSUME_INNER;
                break;
            case STATE_CONSUME_INNER:
            {
                int next = re_end.indexIn(obj_metadata, cursor);
                if (next == -1) {
                    state = STATE_EOS;
                    break;
                }

                const QStringRef value = obj_metadata.midRef(cursor, next - cursor);

                emit objectMetadataDeserialized(current_object_name, value.toString());
                QJsonParseError json_error;
                QJsonDocument json_doc =
                        QJsonDocument::fromJson(value.toUtf8(), &json_error);
                if (json_error.error == QJsonParseError::NoError) {
                    emit objectMetadataDeserializedJson(
                            current_object_name, json_doc);
                }
                cursor = next + re_end.matchedLength();
                state = STATE_SEARCH_BEGIN;
                break;
            }
            default:
                throw std::logic_error("metadataDeserialized(): Invalid state.");
        }
    }
}

//=============================================================================
