/*===========================================================================*\
*                                                                            *
*                              OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
*                                                                            *
\*===========================================================================*/



#ifndef TEXTUREDATA_HH
#define TEXTUREDATA_HH

#include <OpenFlipper/common/perObjectData.hh>
#include "TextureParameters.hh"

#include <QString>
#include <QStringList>
#include <ACG/GL/gl.hh>
#include <vector>
#include <map>
#include <cfloat>


enum TextureType { UNSET         = 1 << 0,
                   /// Texture Coordinates are stored on a per Vertex basis
                   VERTEXBASED   = 1 << 1,
                   /// Texture Coordinates are stored on a per Halfedge basis ( per vertex per face )
                   HALFEDGEBASED = 1 << 2,
                   /// Texture coordinates are autogenerated by OpenGL for Environment map
                   ENVIRONMENT   = 1 << 3,
                   /// MultiTexture Mode
                   MULTITEXTURE  = 1 << 4 };

class Texture {
  public :
    Texture();
    
    // copy constructor
    Texture( const Texture& _tex)
    : parameters(_tex.parameters),
    multiTextureList(_tex.multiTextureList),
    name_(_tex.name_),
    textureImageId_(_tex.textureImageId_),
    visibleName_(_tex.visibleName_),
    filename_(_tex.filename_),
    id_(_tex.id_),
    glName_(_tex.glName_),
    dimension_(_tex.dimension_),
    enabled_(_tex.enabled_),
    hidden_(_tex.hidden_),
    dirty_(_tex.dirty_),
    type_(_tex.type_),
    indexMappingProperty_(_tex.indexMappingProperty_)
    {
    }


    void filename( QString _name ) { filename_ = _name; };
    QString filename() { return filename_; };


    void id( int _id ) { id_ = _id; };
    int id() { return id_; };


    void glName( GLuint _glName ) { glName_ = _glName; };
    GLuint glName() { return glName_; };


    void name( QString _name ) { name_ = _name; };
    QString name() { return name_; };

    void visibleName( QString _name ) { visibleName_ = _name; };
    QString visibleName() { return visibleName_; };


    void dimension( uint _dimension ) { dimension_ = _dimension; };
    uint dimension( ) { return dimension_; };


    void enabled( bool _enabled ) { enabled_ = _enabled; };
    bool enabled() { return enabled_; };
    void enable(){ enabled_ = true; };
    void disable(){ enabled_ = false; };

    void hidden( bool _hidden ) { hidden_ = _hidden; };
    bool hidden() { return hidden_; };
    void hide() { hidden_ = true; };

    void dirty( bool _dirty ) { dirty_ = _dirty; };
    bool dirty() { return dirty_; };
    void clean() { dirty_ = false; };
    void setDirty() { dirty_ = true; };

    void textureImageId( int _id) {textureImageId_ = _id;};
    int textureImageId() {return textureImageId_; };


    void type( TextureType _type ) { type_ = _type; };
    TextureType type( ) { return type_; };


    QString indexMappingProperty() { return indexMappingProperty_; };
    void indexMappingProperty( QString _property ) { indexMappingProperty_ = _property; };


    /// Parameters of the texture
    TexParameters parameters;


    /// If this is a multiTexture, the list will contain all textures for this multi Texture node.
    QStringList multiTextureList;

  private:
    /// Texture Name
    QString name_;

    /// The image used as the texture ( Ids are handled by the ImageStore )
    int textureImageId_;

    /// Name visible in the gui
    QString visibleName_;

    /// Filename of the texture
    QString filename_;

    /// Texture id
    int id_;

    /// glName
    GLuint glName_;

    /// dimension
    uint dimension_;

    /// Status
    bool enabled_;

    /// Hidden flag ( If this texture belongs to a multitexture, it will be hidden in the context menu )
    bool hidden_;

    /// does this texture need an update?
    bool dirty_;

    /// Texture Type
    TextureType type_;

    /** If this is a multiTexture, indexMappingProperty_ will point to the property storing the mapping
     * Defaults to the f:textureindex
    */
    QString indexMappingProperty_;

};

class TextureData : public PerObjectData
{

  public :

      /// Constructor
      TextureData();
      /// Destructor
      ~TextureData();
      
      // copy Operator returning an exact copy of this Object
      virtual PerObjectData* copyPerObjectData( ) {
        
        // Create an object copy (This will call all copy constructors of the included data objects! )
        TextureData* copy = new TextureData(*this);
        
        return copy;
      }


      /// Check if a texture exists
      bool textureExists(QString _textureName);


      /// Check if a texture is enabled
      bool isEnabled(QString _textureName);


      /// Enable a given texture
      bool enableTexture(QString _textureName, bool _exclusive = false);

      /// Disable a given texture
      void disableTexture(QString _textureName);

      /// Add a Texture without file backing
      int addTexture ( QString _textureName , uint _dimension, GLuint _glName ){return addTexture ( _textureName , QString("Invalid") , _dimension, _glName );}
      
      /// Add a Texture
      int addTexture ( QString _textureName , QString _filename , uint _dimension, GLuint _glName );

      /// Add a Texture ( Based on an existing specification )
      int addTexture ( Texture _texture, GLuint _glName );

      /// Adds a new multiTexture ( This texture will only contain a list of enabled textures for multitexturing )
      bool addMultiTexture( QString _textureName );

      /// Stores the given image in the texture information
      bool setImage( QString _textureName , int _id );

      /*
      /// Delete a given texture
      void deleteTexture(QString _textureName);

      /// get parameters of a given texture
      TexParameters textureParameters(QString _textureName);

      /// Set Parameters for a given texture
      void setTextureParameters(QString _textureName, TexParameters _params);
      */

      /// Get the texture object
      Texture& texture(QString _textureName);

      /// Get reference to the texture vector
      std::vector< Texture >& textures();


      /** This map maps all available textures for the object which this
       *  class belongs to to their GLuint. The MeshNode will use this
       *  map to activate one texture for each face.
       *
       */
      std::map< int, GLuint >* textureMap();

    /** This map is used to store the available Textures and map them to their
     *  corresponding properties.
     */
      std::map< int, std::string >* propertyMap();

  private :

    std::map< int, GLuint> textureMap_;
    std::map< int, std::string> propertyMap_;

    /// internal id for the next texture
    int nextInternalID_;


    /// vector containing all textures of an object
    std::vector< Texture > textures_;


    /// Get the index of a given texture
    int getTextureIndex(QString _textureName);

    Texture noTexture;

};

#endif //TEXTUREDATA_HH
