/*===========================================================================*\
*                                                                            *
*                              OpenFlipper                                   *
 *           Copyright (c) 2001-2015, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openflipper.org                            *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenFlipper.                                         *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
*                                                                            *
\*===========================================================================*/



#pragma once


#include <QObject>
#include <QAction>

#include <OpenFlipper/BasePlugin/BaseInterface.hh>
#include <OpenFlipper/BasePlugin/RenderInterface.hh>
#include <OpenFlipper/BasePlugin/LoggingInterface.hh>


#include <ACG/GL/IRenderer.hh>
#include <ACG/GL/FBO.hh>

class DepthPeeling : public QObject, BaseInterface, RenderInterface, LoggingInterface, ACG::IRenderer
{
  Q_OBJECT
    Q_INTERFACES(BaseInterface)
    Q_INTERFACES(RenderInterface)
    Q_INTERFACES(LoggingInterface)

  Q_PLUGIN_METADATA(IID "org.OpenFlipper.Plugins.Plugin-Render-DepthPeeling")

signals:
  // LoggingInterface
  void log(Logtype _type, QString _message);
  void log(QString _message);


public:
  DepthPeeling();
  ~DepthPeeling();

  QString name() { return (QString("DepthPeeling Plugin")); };
  QString description( ) { return (QString(tr("DepthPeeling Rendering Pipeline (Alpha Version!)"))); };


  /// overide addRenderObject function to include OIT check
  void addRenderObject(ACG::RenderObject* _renderObject);


public slots:
  QString version() { return QString("1.0"); };

  QString renderObjectsInfo(bool _outputShaderInfo);

  QAction* optionsAction();

private slots:

  //BaseInterface
  void initializePlugin();
  void exit();

  // RenderInterface
  void render(ACG::GLState* _glState, Viewer::ViewerProperties& _properties);
  QString rendererName() {return QString("Alpha_Version_DepthPeeling");}
  void supportedDrawModes(ACG::SceneGraph::DrawModes::DrawMode& _mode) {_mode = ACG::SceneGraph::DrawModes::DEFAULT;}

  QString checkOpenGL();

  void slotModeChanged( QAction *  );


private:


  /// peel the scene from front to back, one layer per pass
  void renderFrontPeeling(ACG::GLState* _glState, Viewer::ViewerProperties& _properties);

  /// peel the scene with dual depth peeling, two layers per pass
  void renderDualPeeling(ACG::GLState* _glState, Viewer::ViewerProperties& _properties);

  /// mode: 0 -> front to back peeling,  1 -> dual peeling
  int peelMode_;

  bool copyFrontDepth_;

  /// max peel count
  int maxPeelCount_;

  /// blends one depth-layer into the current scene target
  GLSL::Program* peelBlend_;

  /// final copy into back-buffer
  GLSL::Program* peelFinal_;

  /// occlusion query determining end of peeling (last layer)
  GLuint peelQueryID_;


  /// dual depth peeling shaders
  GLSL::Program* peelBlendDual_;
  GLSL::Program* peelFinalDual_;


  /// Collection of framebuffers for each viewport
  struct ViewerResources
  {
    ViewerResources();
    ~ViewerResources();

    // resize textures
    void resize(bool _dualPeeling, unsigned int _width, unsigned int _height);

    /// viewer window width
    unsigned int width_;

    /// viewer window height
    unsigned int height_;


    // single layer depth peeling textures (front to back peeling)
    //  ping-pong buffer for (depth, front) targets
    GLuint singleDepthTex_[2]; // float1: minDepth
    GLuint singleFrontTex_[2]; // rgba: color of front-peeled layer
    GLuint singleBlendTex_;    // rgba: color accumulation buffer

    ACG::FBO* singleFbo_; // targets: {depth0, front0,  depth1, front1, blend}




    // dual depth peeling textures
    //  ping-pong buffer for (depth, front, back) targets
    GLuint dualDepthTex_[2]; // float2: (-minDepth, maxDepth)
    GLuint dualFrontTex_[2]; // rgba: color of front-peeled layer
    GLuint dualBackTex_[2];  // rgba: color of back-peeled layer
    GLuint dualBlendTex_;    // rgb:  color accumulation buffer

    ACG::FBO* dualFbo_; // targets: {depth0, front0, back0,  depth1, front1, back1,  blend}
  };

  /// Allocate framebuffers and load shaders for depth-peeling.
  void initDepthPeeling();

  /// Allocate framebuffers and load shaders for dual-depth-peeling.
  void initDualDepthPeeling();

  /**
  * Stores framebuffer resources for each viewport.
  * Mapping: viewerID -> ViewerResources
  */
  std::map<int, ViewerResources> viewerRes_;




  // debug functions
//   GLSL::Program* dbgProg_;
// 
//   void dbgDrawTex(GLuint _texID);
};
