function [D, A, X, RDA, stat] = DOLPHInRGB(F,Y,fun,proj,params,Xopt,X0,D0,A0)
%function [D, A, X, RDA, stat] = DOLPHInRGB(F,Y,fun,proj,params,Xopt,X0,D0,A0)
%
% This is DOLPHIn, an alternating minimization scheme to simultaneously 
% solve the 2D (generalized) phase retrieval problem and learn a dictionary 
% along with sparse representations for image patches.
%
% Note: This function assumes measurements/image correspond to an RGB image
%       (3 color channels)! Use DOLPHIn.m for grayscale images, see also 
%       testDOLPHIn.m
%
%       D, A and stat will be cell arrays with respective final variables 
%       for each color channel; X and RDA will correspond directly to the
%       RGB reconstructions, so for instance X(:,:,2) is the "Blue" channel
%       component of X. 
%
% INPUT: 
%   F - function implementing the measurement operator; must be of the form
%       F(.,t) such that t=0 applies the op. and t=1 applies its adjoint.
%   Y - the measurements of true image Xopt, i.e., per color channel c:
%       Y{c} = fun(F,Xopt(:,:,c),0)
%   fun - function to evaluate Yhat := fun(F,X,0) = F(X,0) at given X and
%       compute gradient gradX w.r.t. X of the phase retrieval objective 
%       term 0.25*|| Y - Yhat ||_F^2; must be of the form 
%       [Yhat,gradX] = fun(F,X,Y)  (Y is only used in gradient computation)
%   proj - function implementing the projection of X onto some constraints
%       (e.g., proj=@(Z)max(min(Z,1),0).*S projects onto pixel values in 
%       [0,1] w.r.t. a binary support mask S (1 on support, 0 otherwise));
%       must be of the form [projX] = proj(X)
%   params - struct with the algorithmic parameters; except for params.h 
%       and params.w (specifying the image dimensions h x w), all fields 
%       can be left empty to use default values (see setupParams.m)
%
% OPTIONAL INPUT:
%   Xopt - original image (ground truth to be recovered); used to compute
%       some (more) statistics only and can be left empty or omitted
%   X0, D0, A0 - initial image estimate, dictionary and patch representa-
%       tions (columns of A0), resp.; can be left empty or omitted to use
%       defaults [random X0, D0=[ID DCT], A0=argmin||E(X0)-D0*A||_F^2]
%
% OUTPUT:
%   D, A, X - the learned dictionary, (sparse) patch representations and 
%       final image reconstruction
%   RDA - image reconstruction via sparse patch representations; often this
%       gives a better image estimate than the final X
%   stat - cell array, one cell per color channel, with structs containing 
%       various statistics about the DOLPHIn run/sol., most of which are 
%       only evaluated if original image (ground truth) Xopt was provided 
%       by the user. 
%       Fields independent of Xopt are:
%       .params1/.params2 - full set of DOLPHIn parameters in phase 1/2
%       .noPatches - number of patches (= column dimension of A)
%       .tabobj - array containing objective function values throughout the 
%                 iteration process
%       .timeInit - time spent on initialization
%       .timeK1 - time spent in DOLPHIn phase 1 (dictionary kept fixed)
%       .timeK2 - time spent in DOLPHIn phase 2 (dictionary updated)
%       .timeTotal - total runtime of DOLPHIn call
%       .nnzA1  - average number of nonzeros in columns of A after first K1
%            iterations
%       .nnzA   - average number of nonzeros in columns of final A
%       Fields that are only created if Xopt was given are:
%       .mseX0/.snrX0/.psnrX0/.ssimX0 - MSE, SNR, PSNR and SSIM values of
%            initial image estimate X0
%       .mseX1/.snrX1/.psnrX1/.ssimX1 - same for X-estimate after first K1
%            iterations
%       .mseRDA1/.snrRDA1/.psnrRDA1/.ssimRDA1 - same for estimate from
%            patch representations, P_X(R(D*A)), after first K1 iterations
%       .mseX/.snrX/.psnrX/.ssimX - same for final X-estimate
%       .mseRDA/.snrRDA/.psnrRDA/.ssimRDA - same for final P_X(R(D*A))
%   statRGB - struct containing similar statistics w.r.t. the full RGB re-
%       constructions/function call
%
% NOTE: This code requires an installation of the SPAMS package;
% the SPAMS path needs to be set accordingly in setupParams.m
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% This function is part of the DOLPHIn package (version 1.10)
% last modified: 06/07/2016, A. M. Tillmann
%
% You may freely use and modify the code for academic purposes, though we
% would appreciate if you could let us know (particularly should you find 
% a bug); if you use DOLPHIn for your own work, please cite the paper
%
%    "DOLPHIn -- Dictionary Learning for Phase Retrieval",
%    Andreas M. Tillmann, Yonina C. Eldar and Julien Mairal, 2016.
%    http://arxiv.org/abs/1602.02263
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% ========== initialization ===============================================
timer = tic;

recstats = (nargout > 4);

if( nargin < 6 ), Xopt = []; end
if( nargin < 7 ), X0 = []; end
if( nargin < 8 ), D0 = []; end
if( nargin < 9 ), A0 = []; end    

for col=1:3 % separate runs for each color channel
    
    if( isempty(Xopt) ), trueX = []; else trueX = Xopt(:,:,col); end
    if( isempty(X0) ), initX = []; else initX = X0(:,:,col); end
    if( isempty(A0) ), initA = []; else initA = A0(:,:,col); end
    if( isempty(D0) )
        if( col == 1 ), initD = []; else initD = D(:,:,col-1); end
    else
        initD = D0(:,:,col);
    end
    
    if( recstats )
        [D{col}, A{col}, X(:,:,col), RDA(:,:,col), stat{col}] = ...
            DOLPHIn(F,Y{col},fun,proj,params,trueX,initX,initD,initA);
    else
        [D{col}, A{col}, X(:,:,col), RDA(:,:,col)] = ...
            DOLPHIn(F,Y{col},fun,proj,params,trueX,initX,initD,initA);
    end
end

if( recstats )
    statRGB.timeTotal = toc(timer);
    statRGB.statR = stat{1};statRGB.statG = stat{2};statRGB.statB = stat{3};
    statRGB.timeInit = stat{1}.timeInit+stat{2}.timeInit+stat{3}.timeInit;
    statRGB.noPatchesPerChannel = stat{1}.noPatches;
    statRGB.timeK1 = stat{1}.timeK1+stat{2}.timeK1+stat{3}.timeK1;
    statRGB.timeK2 = stat{1}.timeK2+stat{2}.timeK2+stat{3}.timeK2;
    statRGB.nnzA = mean([stat{1}.nnzA,stat{2}.nnzA,stat{3}.nnzA]);
    statRGB.mseX0 = immse(X0,Xopt);
    statRGB.ssimX0 = ssim(X0,Xopt);
    [statRGB.psnrX0,statRGB.snrX0] = psnr(X0,Xopt);
    statRGB.mseX = immse(X,Xopt);
    statRGB.ssimX = ssim(X,Xopt);
    [statRGB.psnrX,statRGB.snrX] = psnr(X,Xopt);
    statRGB.mseRDA = immse(RDA,Xopt);
    statRGB.ssimRDA = ssim(RDA,Xopt);
    [statRGB.psnrRDA,statRGB.snrRDA] = psnr(RDA,Xopt);
end