% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Test script for DOLPHIn (default algorithm variant)
%
% Performs testruns with measurements of the following images and types,
% stores the results in files named meastype_picturename.mat
%
% This script provides the results reported in the DOLPHIn (full) paper 
% w.r.t. the default DOLPHIn variant (cf. Table I):
%
% --- l1-regularization 
% --- random X0
% --- noise-SNR 10dB for Gaussian-type measurements, 20dB for CDPs
% --- oversampling ratios and CDP type as reported, also see below
% --- K1=25, K2=50 (--> 75 iter. for Wirtinger Flow)
% --- disjoint (8x8) patches covering the whole image, resp.
%
% Stored results per image are averaged over 3 runs.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% This script is part of the DOLPHIn package (version 1.10)
% last modified: 06/10/2016, A. M. Tillmann
%
% You may freely use and modify the code for academic purposes, though we
% would appreciate if you could let us know (particularly should you find 
% a bug); if you use DOLPHIn for your own work, please cite the paper
%
%    "DOLPHIn -- Dictionary Learning for Phase Retrieval",
%    Andreas M. Tillmann, Yonina C. Eldar and Julien Mairal, 2016.
%    http://arxiv.org/abs/1602.02263
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

cd ..
clear;

% start test runs:
timer = tic;
format compact;
addpath('tools/'); addpath(cd);
addpath('../code/spams-matlab/build/'); % must be set appropriately

imgpath = 'images/';                    % path to image files
resultspath = 'results/';               % path to store result files 

% grayscale test images:
pictures = {
     'cameraman256.png';          % 1 - 256x256
     'house256.png';              % 2
     'peppers256.png';            % 3
     'lena512.png';               % 4 - 512x512
     'barbara512.png';            % 5
     'boat512.png';               % 6 
     'fingerprint512.png';        % 7
     'mandrill512.png';           % 8
     };

meastypes = {
     'gauss';                     % 1 - |F*X|^2
     'gkronsymm';                 % 2 - |F*X*F'|^2
     'gkron';                     % 3 - |F*X*H'|^2
     'cdp';                       % 4 - |2D-FFT(Mask(i).*X)|^2, i=1:numM
     };
  
% init. parameters for instance generation:
data.oversamplfact =         4; % oversampling factor for non-cdp measurements
data.cdptype       = 'ternary'; % type of diffraction patterns in cdp case
data.numM          =         2; % number of coded diffraction patterns for cdp case 
SNR                =        10; % see instanceGeneration.m for details
savefile           =        []; % see instanceGeneration.m for details

testsPerPic = 3;
%rng(0); % for reproducibility ... (set once in overall testscript file)

proj = @(XX)proj_box(XX);

for ii = 1:length(pictures)
    for jj = 1:length(meastypes)	
        for kk = 1:testsPerPic            
            % some output to see where we are:   
            timer2 = tic;     
            fprintf('Processing %s (%2d of %2d) with %s-type measurements (%d of %d), test %d of %d ... ',pictures{ii},ii,length(pictures),meastypes{jj},jj,length(meastypes),kk,testsPerPic);
            
            if( jj==1 ), SNR = 10; end % gauss variants
            if( jj==4 ), SNR = 20; end % cdp
            
            % generate instance data:
            [Xopt,Y,F,G,H,M] = instanceGenerator([imgpath,pictures{ii}],meastypes{jj},data,savefile,SNR);           
            
            [params.h,params.w] = size(Xopt); params.mY = numel(Y);
            params.verbose = 0; % disable all output for these tests
            
            % random initial image estimate:
            X0 = proj(rand(params.h,params.w)); 
                
            if( jj <= 3 ) % gauss variants
                params.mu = (5e-1)*numel(Y);
                if( jj == 1 ) % gauss
                    params.lambda = numel(Y)*0.105;
                else % gkron(symm)
                    params.lambda = numel(Y)*0.210;
                end
            else % cdp
                params.mu = (5e-2)*numel(Y);
                params.lambda = (3e-3)*numel(Y);
            end
            params = setupParams(params);
                
            % run DOLPHIn: (D0 and A0 init. within DOLPHIn function)
            [D,A,X,RDA,stats] = DOLPHIn(F,Y,@fun_general_op,proj,params,Xopt,X0);

            stat.DOLPHInStats{kk} = stats;
            
            % add some instance information to statistics struct:
            stat.meastype = meastypes{jj};
            stat.imgname  = pictures{ii};
            stat.Y_snr    = SNR;
            stat.instGenData = data;
            
            % Wirt. Flow run: 
            timer3 = tic;
            initstepsize = stats.params1.gammaX;%1;
            Xwf = WF_RealCons(F,Y,@fun_general_op,proj,params.h,params.w,params.K1+params.K2,initstepsize,X0,0);
            
            stat.timeWF(kk) = toc(timer3);
            stat.mseXwf(kk) = mse(Xwf,Xopt);
            stat.ssimXwf(kk) = ssim(Xwf,Xopt);
            [stat.psnrXwf(kk),stat.snrXwf(kk)] = psnr(Xwf,Xopt);
            
            fprintf('done! (elapsed time: %.2fs)\n',toc(timer2));           
           
            % reset parameters:
            clear params;
        end
        
        % store results:  
        dots = strfind(pictures{ii},'.'); % last dot in image filename separate name and extension 
        outputfile = [resultspath,'VariantDefault_',meastypes{jj},'_',pictures{ii}(1:dots(end)-1),'.mat'];
        
        save(outputfile,'stat'); % only store 'stat', to save space
        
    end
end
fprintf('VariantDefault -- All done! Total time elapsed: %.2fs\n',toc(timer));


% compute result statistics
%addpath('testscripts');
%ICASSPpaper__script_evaluateResults

cd scripts