% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Test script for DOLPHIn (default algorithm variant)
%
% Performs testruns with measurements of the following images and types,
% and several parameter variations; stores the results in files named 
% VariantDefault_meastype_picturename_PARLIST.mat; PARLIST explained below.
%
% This script provides additional results partly reported in the DOLPHIn 
% (full) paper w.r.t. the default DOLPHIn variant:
%
% --- l1-regularization 
% --- random X0
% --- noise-SNR 10dB for Gaussian-type measurements, 20dB for CDPs
% --- oversampling ratios and CDP type as reported, also see below
% --- K1=25, K2=50 (--> 75 iter. for Wirtinger Flow)
% --- disjoint (8x8) patches covering the whole image, resp.
%
% Here, variations of parameter settings are tested, which are reflected 
% in the suffix PARLIST of each output file created. PARLIST takes the form
% K1_K2_a_b_c_d, where 
%   K1,K2: iteration limits for DOLPHIn phases, see setupParams.m
%   a: 0 or 1, denotes whether A is kept fixed for the first K1 iter. (1) 
%      or not (0)
%   b: number of (F)ISTA iterations for A-update
%   c: number of block-coordinate descent iterations for D-update
%   d: number of proj. gradient steps for X-update
%
% The following combinations are tested:
%  0_75_0_1_1_1
%  75_0_0_1_1_1
%  25_50_a_b_c_d for a in {0,1} and b,c in {1,3,5} and d in {1,3} 
%       (except a=0,b=c=d=1; this case is the usual default and tested with 
%        testscript_DOLPHIn_VariantDefault.m)
%
% For the script to reproduce results from Table I in the paper, see 
% testscript_DOLPHIn_VariantDefault.m
%
% Stored results per image are averaged over 3 runs.
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% This script is part of the DOLPHIn package (version 1.10)
% last modified: 06/16/2016, A. M. Tillmann
%
% You may freely use and modify the code for academic purposes, though we
% would appreciate if you could let us know (particularly should you find 
% a bug); if you use DOLPHIn for your own work, please cite the paper
%
%    "DOLPHIn -- Dictionary Learning for Phase Retrieval",
%    Andreas M. Tillmann, Yonina C. Eldar and Julien Mairal, 2016.
%    http://arxiv.org/abs/1602.02263
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

cd ..
clear;

% start test runs:
timer = tic;
format compact;
addpath('tools/'); addpath(cd);
addpath('../code/spams-matlab/build/'); % must be set appropriately

imgpath = 'images/';                    % path to image files
resultspath = 'results/';               % path to store result files 

% grayscale test images:
pictures = {
     'cameraman256.png';          % 1 - 256x256
     'house256.png';              % 2
     'peppers256.png';            % 3
     'lena512.png';               % 4 - 512x512
     'barbara512.png';            % 5
     'boat512.png';               % 6 
     'fingerprint512.png';        % 7
     'mandrill512.png';           % 8
     };

meastypes = {
     'gauss';                     % 1 - |F*X|^2
     'gkronsymm';                 % 2 - |F*X*F'|^2
     'gkron';                     % 3 - |F*X*H'|^2
     'cdp';                       % 4 - |2D-FFT(Mask(i).*X)|^2, i=1:numM
     };
  
% init. parameters for instance generation:
data.oversamplfact =         4; % oversampling factor for non-cdp measurements
data.cdptype       = 'ternary'; % type of diffraction patterns in cdp case
data.numM          =         2; % number of coded diffraction patterns for cdp case 
SNR                =        10; % see instanceGeneration.m for details
savefile           =        []; % see instanceGeneration.m for details

testsPerPic = 3;
%rng(0); % for reproducibility ... (set once in overall testscript file)

proj = @(XX)proj_box(XX);


%% 0_75_0_1_1_1 : (updating D (and A) right from the beginning)
% rng(0); % so that results are directly comparable to those for default settings (cf. testscript_DOLPHIn_VariantDefault.m)
% for ii = 1:length(pictures)
%     for jj = 1:length(meastypes)	
%         for kk = 1:testsPerPic            
%             % some output to see where we are:   
%             timer2 = tic;     
%             fprintf('0_75_0_1_1_1: Processing %s (%2d of %2d) with %s-type measurements (%d of %d), test %d of %d ... ',pictures{ii},ii,length(pictures),meastypes{jj},jj,length(meastypes),kk,testsPerPic);
%             
%             if( jj==1 ), SNR = 10; end % gauss variants
%             if( jj==4 ), SNR = 20; end % cdp
%             
%             % generate instance data:
%             [Xopt,Y,F,G,H,M] = instanceGenerator([imgpath,pictures{ii}],meastypes{jj},data,savefile,SNR);           
%             
%             [params.h,params.w] = size(Xopt); params.mY = numel(Y);
%             params.verbose = 0; % disable all output for these tests
%             
%             params.K1 = 0; params.K2 = 75;
%             
%             % random initial image estimate:
%             X0 = proj(rand(params.h,params.w)); 
%                 
%             if( jj <= 3 ) % gauss variants
%                 params.mu = (5e-1)*numel(Y);
%                 if( jj == 1 ) % gauss
%                     params.lambda = numel(Y)*0.105;
%                 else % gkron(symm)
%                     params.lambda = numel(Y)*0.210;
%                 end
%             else % cdp
%                 params.mu = (5e-2)*numel(Y);
%                 params.lambda = (3e-3)*numel(Y);
%             end
%             params = setupParams(params);
%                 
%             % run DOLPHIn: (D0 and A0 init. within DOLPHIn function)
%             [D,A,X,RDA,stats] = DOLPHIn(F,Y,@fun_general_op,proj,params,Xopt,X0);
% 
%             stat.DOLPHInStats{kk} = stats;
%             
%             % add some instance information to statistics struct:
%             stat.meastype = meastypes{jj};
%             stat.imgname  = pictures{ii};
%             stat.Y_snr    = SNR;
%             stat.instGenData = data;
%             
%             % Wirt. Flow run: 
%             timer3 = tic;
%             initstepsize = stats.params1.gammaX;%1;
%             Xwf = WF_RealCons(F,Y,@fun_general_op,proj,params.h,params.w,params.K1+params.K2,initstepsize,X0,0);
%             
%             stat.timeWF(kk) = toc(timer3);
%             stat.mseXwf(kk) = mse(Xwf,Xopt);
%             stat.ssimXwf(kk) = ssim(Xwf,Xopt);
%             [stat.psnrXwf(kk),stat.snrXwf(kk)] = psnr(Xwf,Xopt);
%             
%             fprintf('done! (elapsed time: %.2fs)\n',toc(timer2));           
%            
%             % reset parameters:
%             clear params;
%         end
%         
%         % store results:  
%         dots = strfind(pictures{ii},'.'); % last dot in image filename separate name and extension 
%         outputfile = [resultspath,'VariantDefault_',meastypes{jj},'_',pictures{ii}(1:dots(end)-1),'_0_75_0_1_1_1','.mat'];
%         
%         save(outputfile,'stat'); % only store 'stat', to save space
%         
%     end
% end

%% 75_0_0_1_1_1 : (corresponds to WF w/ sparse-patch-regul. w/ fixed D=[DCT,ID])
% rng(0); % so that results are directly comparable to those for default settings (cf. testscript_DOLPHIn_VariantDefault.m)
% for ii = 1:length(pictures)
%     for jj = 1:length(meastypes)	
%         for kk = 1:testsPerPic            
%             % some output to see where we are:   
%             timer2 = tic;     
%             fprintf('75_0_0_1_1_1: Processing %s (%2d of %2d) with %s-type measurements (%d of %d), test %d of %d ... ',pictures{ii},ii,length(pictures),meastypes{jj},jj,length(meastypes),kk,testsPerPic);
%             
%             if( jj==1 ), SNR = 10; end % gauss variants
%             if( jj==4 ), SNR = 20; end % cdp
%             
%             % generate instance data:
%             [Xopt,Y,F,G,H,M] = instanceGenerator([imgpath,pictures{ii}],meastypes{jj},data,savefile,SNR);           
%             
%             [params.h,params.w] = size(Xopt); params.mY = numel(Y);
%             params.verbose = 0; % disable all output for these tests
%             
%             params.K1 = 75; params.K2 = 0;
%             
%             % random initial image estimate:
%             X0 = proj(rand(params.h,params.w)); 
%                 
%             if( jj <= 3 ) % gauss variants
%                 params.mu = (5e-1)*numel(Y);
%                 if( jj == 1 ) % gauss
%                     params.lambda = numel(Y)*0.105;
%                 else % gkron(symm)
%                     params.lambda = numel(Y)*0.210;
%                 end
%             else % cdp
%                 params.mu = (5e-2)*numel(Y);
%                 params.lambda = (3e-3)*numel(Y);
%             end
%             params = setupParams(params);
%                 
%             % run DOLPHIn: (D0 and A0 init. within DOLPHIn function)
%             [D,A,X,RDA,stats] = DOLPHIn(F,Y,@fun_general_op,proj,params,Xopt,X0);
% 
%             stat.DOLPHInStats{kk} = stats;
%             
%             % add some instance information to statistics struct:
%             stat.meastype = meastypes{jj};
%             stat.imgname  = pictures{ii};
%             stat.Y_snr    = SNR;
%             stat.instGenData = data;
%             
%             % Wirt. Flow run: 
%             timer3 = tic;
%             initstepsize = stats.params1.gammaX;%1;
%             Xwf = WF_RealCons(F,Y,@fun_general_op,proj,params.h,params.w,params.K1+params.K2,initstepsize,X0,0);
%             
%             stat.timeWF(kk) = toc(timer3);
%             stat.mseXwf(kk) = mse(Xwf,Xopt);
%             stat.ssimXwf(kk) = ssim(Xwf,Xopt);
%             [stat.psnrXwf(kk),stat.snrXwf(kk)] = psnr(Xwf,Xopt);
%             
%             fprintf('done! (elapsed time: %.2fs)\n',toc(timer2));           
%            
%             % reset parameters:
%             clear params;
%         end
%         
%         % store results:  
%         dots = strfind(pictures{ii},'.'); % last dot in image filename separate name and extension 
%         outputfile = [resultspath,'VariantDefault_',meastypes{jj},'_',pictures{ii}(1:dots(end)-1),'_75_0_0_1_1_1','.mat'];
%         
%         save(outputfile,'stat'); % only store 'stat', to save space
%         
%     end
% end

%% 25_50_a_b_c_d :
for a = 0:1
    for b = (1+(1-a)*2):2:5
        for c = 1:2:5
            for d = 1:2:3
                if( a+b+c+d > 3 ) % exludes the case a=0,b=c=d=1 (overall DOLPHIn default)
                    rng(0); % so that results are directly comparable to those for default settings (cf. testscript_DOLPHIn_VariantDefault.m)
                    for ii = 1:length(pictures)
                        for jj = 1:length(meastypes)
                            for kk = 1:testsPerPic
                                % some output to see where we are:
                                timer2 = tic;
                                fprintf('25_50_%d_%d_%d_%d: Processing %s (%2d of %2d) with %s-type measurements (%d of %d), test %d of %d ... ',a,b,c,d,pictures{ii},ii,length(pictures),meastypes{jj},jj,length(meastypes),kk,testsPerPic);
                                
                                if( jj==1 ), SNR = 10; end % gauss variants
                                if( jj==4 ), SNR = 20; end % cdp
                                
                                % generate instance data:
                                [Xopt,Y,F,G,H,M] = instanceGenerator([imgpath,pictures{ii}],meastypes{jj},data,savefile,SNR);
                                
                                [params.h,params.w] = size(Xopt); params.mY = numel(Y);
                                params.verbose = 0; % disable all output for these tests
                               
                                params.AfixK1 = a;
                                params.L = -b;
                                params.BCDiter = c;
                                params.Xiter = d;
                                
                                % random initial image estimate:
                                X0 = proj(rand(params.h,params.w));
                                
                                if( jj <= 3 ) % gauss variants
                                    params.mu = (5e-1)*numel(Y);
                                    if( jj == 1 ) % gauss
                                        params.lambda = numel(Y)*0.105;
                                    else % gkron(symm)
                                        params.lambda = numel(Y)*0.210;
                                    end
                                else % cdp
                                    params.mu = (5e-2)*numel(Y);
                                    params.lambda = (3e-3)*numel(Y);
                                end
                                params = setupParams(params);
                                
                                % run DOLPHIn: (D0 and A0 init. within DOLPHIn function)
                                [D,A,X,RDA,stats] = DOLPHIn(F,Y,@fun_general_op,proj,params,Xopt,X0);
                                
                                stat.DOLPHInStats{kk} = stats;
                                
                                % add some instance information to statistics struct:
                                stat.meastype = meastypes{jj};
                                stat.imgname  = pictures{ii};
                                stat.Y_snr    = SNR;
                                stat.instGenData = data;
                                
                                % Wirt. Flow run:
                                timer3 = tic;
                                initstepsize = stats.params1.gammaX;%1;
                                Xwf = WF_RealCons(F,Y,@fun_general_op,proj,params.h,params.w,params.K1+params.K2,initstepsize,X0,0);
                                
                                stat.timeWF(kk) = toc(timer3);
                                stat.mseXwf(kk) = mse(Xwf,Xopt);
                                stat.ssimXwf(kk) = ssim(Xwf,Xopt);
                                [stat.psnrXwf(kk),stat.snrXwf(kk)] = psnr(Xwf,Xopt);
                                
                                fprintf('done! (elapsed time: %.2fs)\n',toc(timer2));
                                
                                % reset parameters:
                                clear params;
                            end
                            
                            % store results:
                            dots = strfind(pictures{ii},'.'); % last dot in image filename separate name and extension
                            outputfile = [resultspath,'VariantDefault_',meastypes{jj},'_',pictures{ii}(1:dots(end)-1),'_25_50_',sprintf('%d_%d_%d_%d',a,b,c,d),'.mat'];
                            
                            save(outputfile,'stat'); % only store 'stat', to save space
                            
                        end
                    end

                end
            end
        end
    end
end


fprintf('VariantDefault variations -- All done! Total time elapsed: %.2fs\n',toc(timer));


% compute result statistics
%addpath('testscripts');
%ICASSPpaper__script_evaluateResults

cd scripts