function params = setupParams(par,spamspath)
%function params = setupParams(par,spamspath)
%
% initializes parameter struct for DOLPHIn method;
% keeps user settings and sets remaining parameters to default values.
%
% INPUT:
% par - user settings, valid fields: see parameter list below
%       -------------------------------------------------------------------
%       !!! par.h, par.w (orig. image dimensions) MUST NOT BE EMPTY !!!
%       !!! par.mY (number of measurements) MUST NOT BE EMPTY !!!
%       -------------------------------------------------------------------
%
% OPTIONAL INPUT:
% spamspath - path to SPAMS installation folder; if left out, it is assumed
%             that SPAMS is installed in <current-folder>/spams-matlab
%
% OUTPUT:
% params - struct containing all algorithmic/model parameters for DOLPHIn
%
% COMPLETE LIST OF PARAMETERS:
% params.h/w     = image dimensions (h x w),
%                  !!! MUST BE GIVEN (as par.h, par.w) !!!
% params.mY      = number of measurements !!! MUST BE GIVEN (as par.mY) !!!
% params.L       = controls how repres. vectors (columns of A) are updated:
%                  L<0 -> solve l1-reg. LS obj. with (F)ISTA, 
%                         -L=|L| is taken as iteration limit
%                  L=0 -> solve l1-reg. LS obj. with homotopy method, 
%                  L>0 -> solve sparsity-constrained LS obj. with OMP, 
%                         L is taken as sparsity level/constraint
% params.lamba   = regularization parameter for l1-reg. objective function
% params.mu      = penalty parameter for patch-fit term in objective fct.
% params.gammaX  = initial step size parameter for X-update (grad. desc.)
% params.K1/K2   = iteration limits of DOLPHIn phases without (K1) and with
%                  (K2) dictionary updates
% params.AfixK1  = toggle: 1 - do not update A the first K1 iterations, 
%                  0 - always update A
% params.adjustA = toggle: adjust A w.r.t. scaling of D-columns (1) or not
%                  (0) within the D-update part (see DOLPHIn.m for details)
% params.adjFinA = toggle: 1 - update A once more after final iteration, if
%                  dictionary was updated last, 0 - do not adjust final A
% params.overlap = divide image into complete set of overlapping (1) or
%                  non-overlapping (0) patches (to be coded sparsely)
% params.verbose = toggle output: 0 - no output, 1 - iteration log, 
%                  2 - additionally plot progress of reconstructions and
%                  objective function values
% params.s1/s2   = size of image patches (s1 x s2)
% params.BCDiter = number of block-coordinate desc. iterations for D-update
% params.FISTApar= parameter struct for (F)ISTA method (used if params.L<0),
%                  contains (fixed) .loss and .regul ('square' and 'l1', 
%                  resp., for l1-reg. LS) and options .lambda (regulariza-
%                  tion parameter), .max_it (iter. limit), .fixed_step, .L0
%                  (if fixed_step=true, deactive linesearch and use con-
%                  stant L0 instead of Lipschitz constant estimates), .ista 
%                  (set to true to use ISTA instead of FISTA), and 
%                  .numThreads (# of parallel threads to use; -1 -> automa-
%                  tically use all available CPUs/cores, default)
% params.LASSOpar= parameter struct for homotopy method (used only if 
%                  params.L=0), contains .lambda (regularization parameter) 
%                  and .numThreads
% params.OMPpar  = parameter struct for OMP (used if params.L>0),
%                  contains .L (sparsity bound), .eps (deviation tolerance)
%                  and .numThreads
%
% (for more possible parameters for OMP/LASSO/(F)ISTA, see SPAMS package)
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% This function is part of the DOLPHIn package (version 1.10)
% last modified: 06/16/2016, A. M. Tillmann
%
% You may freely use and modify the code for academic purposes, though we
% would appreciate if you could let us know (particularly should you find 
% a bug); if you use DOLPHIn for your own work, please cite the paper
%
%    "DOLPHIn -- Dictionary Learning for Phase Retrieval",
%    Andreas M. Tillmann, Yonina C. Eldar and Julien Mairal, 2016.
%    http://arxiv.org/abs/1602.02263
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% default settings:
params.h       = par.h; 
params.w       = par.w;
params.mY      = par.mY;

params.L       = -1; % 1 (F)ISTA iteration
params.lambda  = 3e-3 *params.mY; 
params.mu      = 5e-2 *params.mY; 
% =========================================================================
% remark: good choices for lambda and mu depend on the measurement model 
%         and noise level (and to a lesser extent, other model parameters); 
%         for some suggestions, please check our paper (see ref. above)!
%         [values set here correspond to CDP experiments from the paper]
% =========================================================================
params.gammaX  = 1;  % default from paper is set in DOLPHIn.m
params.K1      = 25;
params.K2      = 50;
params.AfixK1  = 0; 
params.adjustA = 0;
params.adjFinA = 0;
params.overlap = 0;
params.verbose = 0;
params.s1      = 8;
params.s2      = 8;
params.Xiter   = 1; % number of projectd gradient descent steps performed during X-update
params.BCDiter = 1; % larger BCDiter, in particular combined with larger |L| (say, 5 each), can give improvements
params.FISTApar.loss = 'square';
params.FISTApar.regul = 'l1';
params.FISTApar.lambda = params.lambda/params.mu;  
params.FISTApar.max_it = -params.L;
params.FISTApar.fixed_step = 0;
params.FISTApar.L0 = 0.1; % FISTA default corresp. to fixed_step=0
params.FISTApar.ista = 1; % our default: 1 iter. of ISTA
params.FISTApar.numThreads = -1;
params.OMPpar.L = params.L;
params.OMPpar.eps = 1e-1;
params.OMPpar.numThreads = -1;
params.LASSOpar.lambda = params.lambda/params.mu;
params.LASSOpar.numThreads = -1;

% copy user settings:
if( isfield(par,'L') ),      params.L = par.L; params.OMPpar.L = params.L; params.FISTApar.max_it = -params.L; end
if( isfield(par,'lambda') ), params.lambda = par.lambda; params.LASSOpar.lambda = params.lambda/params.mu; params.FISTApar.lambda = params.lambda/params.mu; end
if( isfield(par,'mu') )
    params.mu = par.mu; params.LASSOpar.lambda = params.lambda/params.mu; 
    params.FISTApar.lambda = params.lambda/params.mu; 
elseif( params.L > 0 ) % mu-default for experiments from paper for sparsity-constr. DOLPHIn:
    params.mu = 5e-3 *params.mY; params.LASSOpar.lambda = params.lambda/params.mu; 
    params.FISTApar.lambda = params.lambda/params.mu; 
end
if( isfield(par,'gammaX') ), params.gammaX = par.gammaX; end
if( isfield(par,'K1') ),     params.K1 = par.K1; end
if( isfield(par,'K2') )
    params.K2 = par.K2; 
elseif( params.L > 0 ) % sparsity-constr. DOLPHIn: default is K2=K1 
    params.K2 = params.K1; 
    params.lambda = 0; % set lambda to zero so that obj. is computed correctly within DOLPHIn:
end
if( isfield(par,'AfixK1') ),  params.AfixK1 = par.AfixK1; end
if( isfield(par,'adjustA') ), params.adjustA = par.adjustA; end
if( isfield(par,'adjFinA') ), params.adjFinA = par.adjFinA; end
if( isfield(par,'overlap') ), params.overlap = par.overlap; end
if( isfield(par,'verbose') ), params.verbose = par.verbose; end
if( isfield(par,'s1') ),      params.s1 = par.s1; end
if( isfield(par,'s2') ),      params.s2 = par.s2; end
if( isfield(par,'Xiter') ),   params.Xiter = par.Xiter; end
if( isfield(par,'BCDiter') ), params.BCDiter = par.BCDiter; end
if( isfield(par,'FISTApar') )
    if( isfield(par.FISTApar,'loss') ), params.FISTApar.loss = par.FISTApar.loss; end
    if( isfield(par.FISTApar,'regul') ), params.FISTApar.regul = par.FISTApar.regul; end
    %if( isfield(par.FISTApar,'lambda') ), params.FISTApar.lambda = par.FISTApar.lambda; end
    %if( isfield(par.FISTApar,'max_it') ), params.FISTApar.max_it = par.FISTApar.max_it; end
    if( isfield(par.FISTApar,'fixed_step') ), params.FISTApar.fixed_step = par.FISTApar.fixed_step; end
    if( isfield(par.FISTApar,'L0') ), params.FISTApar.L0 = par.FISTApar.L0; end
    if( isfield(par.FISTApar,'ista') ), params.FISTApar.ista = par.FISTApar.ista; end
    if( isfield(par.FISTApar,'numThreads') ), params.FISTApar.numThreads = par.FISTApar.numThreads; end
end
if( isfield(par,'OMPpar') )
    %if( isfield(par.OMPpar,'L') ), params.OMPpar.L = par.OMPpar.L; end
    if( isfield(par.OMPpar,'eps') ), params.OMPpar.eps = par.OMPpar.eps; end
    if( isfield(par.OMPpar,'numThreads') ), params.OMPpar.numThreads = par.OMPpar.numThreads; end
end
if( isfield(par,'LASSOpar') )
    %if( isfield(par.LASSOpar,'lambda') ), params.LASSOpar.lambda = par.LASSOpar.lambda; end
    if( isfield(par.LASSOpar,'numThreads') ), params.LASSOpar.numThreads = par.LASSOpar.numThreads; end
end

% also, add folder with auxiliary functions to path:
addpath('tools/');
if( nargin < 2 ) % assuming spams is installed here:
    addpath('../code/spams-matlab/build/');
else
    addpath([spamspath,'build/']);
end
