function [X,tabobj] = WF_RealCons(F,Y,fun,proj,M,N,iter,eta,X0,verbose)
%function [X,tabobj] = WF_RealCons(F,Y,fun,proj,M,N,iter,eta,X0,verbose)
%
% Implementation of Wirtinger Flow / projected gradient descent
% for generalized (2D) phase retrieval problem: 
% 
%      min   0.25*|| fun( F(X) ) - Y||_F^2  
%      s.t.  X MxN real-valued and in some constraint set S
%
% variant with stepsize line search aiming at monotonic objective decrease
%
% Input:
%
% F - linear measurement operator; must be a function handle of form F(.,t) 
%     such that F(.,0) applies F and F(.,1) its adjoint
% Y - given measurements of original image Xopt (Y=fun(F(Xopt,0)))
% fun - function that evaluates measurement model fun(F(.,0)) 
%     at given X and computes the gradient of the data-fit term
%     0.25*||Y-fun(F(X,0))||_F^2; must be of the form 
%     [Yhat,grad] = fun(F,X,Y) with F the measurement operator
%     and Y the original measurements (only needed if grad is to
%     be computed)
% proj - function implementing the (Euclidean) projection onto 
%     the constraint set S of a given X
% M,N - size of original image/solution (MxN matrix) 
%
% Optional input:
%
% iter - the number of iterations to perform (default: 75)
% eta - initial step size parameter (default: 2)
% X0 - initial image estimate (default: 10 iter.s of power method)
% verbose - 0: no output, 1 (default): iter. log/algo. progress
%
% Output:
% 
% X - the computed phase retrieval solution estimate
% tabobj - array containing objective function values over iter.s
%     
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% This function is part of the DOLPHIn package (version 1.10)
% last modified: 06/06/2016, A. M. Tillmann
%
% You may freely use and modify the code for academic purposes, though we
% would appreciate if you could let us know (particularly should you find 
% a bug); if you use DOLPHIn for your own work, please cite the paper
%
%    "DOLPHIn -- Dictionary Learning for Phase Retrieval",
%    Andreas M. Tillmann, Yonina C. Eldar and Julien Mairal, 2016.
%    http://arxiv.org/abs/1602.02263
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%% set defaults, if needed: %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if( nargin < 7 ),  iter    = 75; end
if( nargin < 8 ),  eta     = 2;  end
if( nargin < 9 ),  X0      = WFinitX0(F,Y,[M,N]); end
if( nargin < 10 ), verbose = 1; end

% %%%%%%%%%% initialization %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
X = X0;
objval = @(YYhat)0.25*sum(abs(YYhat(:)-Y(:)).^2);
obj = objval(fun(F,X0,Y)); 
if( verbose )
    fprintf('Init. obj. val.: %e\n',obj);
end
tabobj = obj;

% %%%%%%%%%% main loop %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for ii=1:iter
   [~, grad] = fun(F,X,Y);  
   oldobj = obj;
   Xold = X;
   count = 1;
   while(true)
      X = proj(Xold - eta*grad);

      Yhat = fun(F,X,Y);
      obj = objval(Yhat);

      if( oldobj >= obj )
         break;
      end
      eta = eta/2;
      count = count+1;
      if( count >= 100 )
         break;
      end
   end
   eta = eta*2;
   if( count == 100 )
      break;
   end   
   if( verbose )
      fprintf('Iter %4d - Stepsize used: %.3e - New objective: %.3e\n',ii,eta/2,obj);
      tabobj = [tabobj obj];
   end
end
