function [X,tabobj] = WF_RealCons_WFsteps(G,Y,fun,proj,M,N,iter,eta,X0,verbose)
%function [X,tabobj] = WF_RealCons_WFsteps(G,Y,fun,proj,M,N,iter,eta,X0,verbose)
%
% Implementation of Wirtinger Flow / projected gradient descent
% for generalized (2D) phase retrieval problem: 
% 
%     min   0.25*|| F( G(X) ) - Y||_F^2  
%     s.t.  X MxN real-valued and in some constraint set S
%
% variant with step size control as suggested in Candes et al. WF paper
%
% Input:
%
% G - linear measurement operator; must be a function handle of form G(.,t) 
%     such that G(.,0) applies G and G(.,1) its adjoint
% Y - given measurements of original image Xopt (Y=F(G(Xopt,0)))
% fun - function that evaluates measurement model F(G(.,0)) 
%     at given X and computes the gradient of the data-fit term
%     0.25*||Y-F(G(X,0))||_frob^2; must be of the form 
%     [Yhat,grad] = fun(G,X,Y) with G the measurement operator
%     and Y the original measurements (only needed if gradF is to
%     be computed)
% proj - function implementing the (Euclidean) projection onto 
%     the constraint set S of a given X
% M,N - size of original image/solution (MxN matrix) 
%
% Optional input:
%
% iter - the number of iterations to perform (default: 75)
% eta - initial step size parameter (default: 2)
% X0 - initial image estimate (default: 10 iter.s of power method)
% verbose - 0: no output, 1 (default): iter. log/algo. progress
%
% Output:
% 
% X - the computed phase retrieval solution estimate
% tabobj - array containing objective function values over iter.s
%     
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% This function is part of the DOLPHIn package (version 1.10)
% last modified: 02/06/2016, A. M. Tillmann
%
% You may freely use and modify the code for academic purposes, though we
% would appreciate if you could let us know (particularly should you find 
% a bug); if you use DOLPHIn for your own work, please cite the paper
%
%    "DOLPHIn -- Dictionary Learning for Phase Retrieval",
%    Andreas M. Tillmann, Yonina C. Eldar and Julien Mairal, 2016.
%    http://arxiv.org/abs/1602.02263
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% %%%%%%%%%% set defaults, if needed: %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if( nargin < 7 ),  iter    = 75; end
if( nargin < 8 ),  eta     = 2;  end
if( nargin < 9 ),  X0      = WFinitX0(G,Y,[M,N]); end
if( nargin < 10 ), verbose = 1; end

% %%%%%%%%%% initialization %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
X = X0;
objval = @(YYhat)0.25*sum(abs(YYhat(:)-Y(:)).^2);
[Yhat,grad] = fun(G,X0,Y);
obj = objval(Yhat); 
if( verbose )
    fprintf('Init. obj. val.: %e\n',obj);
end
tabobj = obj;

normX0sq = sum(sum(X0.^2)); % Candes et al. suggest div. step size by this 

% %%%%%%%%%% main loop %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
for ii=1:iter
   eta = min(1-exp(-(ii+1)/330),0.4); % Candes et al. step size control
   
   X = proj(X - eta*grad/normX0sq); % Candes et al: div. by normX0sq
   
   [Yhat,grad] = fun(G,X,Y);
   obj = objval(Yhat);
   if( verbose )
      fprintf('Iter %4d - Stepsize used: %.3e - New objective: %.3e\n',ii,eta,obj);
      tabobj = [tabobj obj];
   end
end
