function X0 = WFinitX0(G,Y,sizeX,proj,iter,scalefact,scaleexp)
%function X0 = WFinitX0(G,Y,sizeX,proj,iter,scalefact,scaleexp)
%
% Power iterations for initial image estimate, 
%     modified variant of the scheme from Candes et al. (2014),
%     "Phase Retrieval via Wirtinger Flow: Theory and Algorithms"
%
% Input:
%
% G - lin. measurement operator, G(.,0) applies the op., G(.,1) its adjoint
% Y - measurements of original image (Y = fun_general/lin_op(G,Xopt,0) )
% sizeX - [h,w] (or [h w]) - the size of the image (h=height, w=width)
%
% Optional input:
%
% proj - function to project X onto image constraints [default: proj_box]
% iter - number of power iterations to perform [default: 10]
% scalenormest - factor for scaling power iterations outcome [default: 0.2*sqrt(numel(Y))]
% scaleexp - exponent for scaling power iterations outcome [default: 0.2]
%      (X0-values are in (0,1), so taking powers <1 increases the entries)
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% This function is part of the DOLPHIn package (version 1.10)
% last modified: 02/06/2016, A. M. Tillmann
%
% You may freely use and modify the code for academic purposes, though we
% would appreciate if you could let us know (particularly should you find 
% a bug); if you use DOLPHIn for your own work, please cite the paper
%
%    "DOLPHIn -- Dictionary Learning for Phase Retrieval",
%    Andreas M. Tillmann, Yonina C. Eldar and Julien Mairal, 2016.
%    http://arxiv.org/abs/1602.02263
%
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% set defaults, if needed:
if( nargin < 4 ), proj = @proj_box; end
if( nargin < 5 ), iter = 10; end
if( nargin < 6 ), scalefact = 0.2*sqrt(numel(Y)); end
if( nargin < 7 ), scaleexp = 0.2; end

ignoreReal01 = 1;

% initial guess:
if( ignoreReal01 )
    X0 = randn(sizeX);
else
    X0 = rand(sizeX);
end

X0 = X0/norm(X0,'fro'); 

% power iterations:
for ii = 1:iter
    X0 = G(Y.*G(X0,0),1);

    if( ~ignoreReal01 )
        X0 = max(0,real(X0));
    end
        
    X0 = X0/norm(X0,'fro');
end

% some scaling parameters (to scale computed approx. eigenvector) ...
if( isreal(Y) )
    normest = sqrt(sum(Y(:))/numel(Y)); % est. norm 
else
    normest = sqrt(sum(abs(Y(:)))/numel(Y)); % est. norm
end
        
% apply scaling and perform final projection:
if( ignoreReal01 )
    X0 = normest * abs(X0); % what Candes et al. do; alt.: max(0,real(X0))*normest  
else
    X0 = proj( (scalefact * normest * real(X0)).^scaleexp );
end
