#include "SummerSchoolSmootherPlugin.hh"
#include "OpenFlipper/BasePlugin/PluginFunctions.hh"
#include <OpenMesh/Core/Utils/PropertyManager.hh>
//-----------------------------------------------------------------------------

void SummerSchoolSmootherPlugin::computeNewPositions(TriMeshObject* _meshObject)
{
    // Compute new positions using Laplace or Laplace^2 smoothing

    TriMesh& mesh = *_meshObject->mesh();
    auto newPos = OpenMesh::PropertyManager<OpenMesh::VPropHandleT<TriMesh::Point>, TriMesh>::createIfNotExists(mesh, "newPos");
    auto edgeWeight
        = OpenMesh::PropertyManager<OpenMesh::EPropHandleT<double>, TriMesh>::createIfNotExists(mesh, "edgeWeight");


    for (auto vh : mesh.vertices())
    {
        // Do not smooth locked vertices (e.g. boundary)
        if (!mesh.status(vh).locked())
        {
            TriMesh::Point updateVector = TriMesh::Point(0, 0, 0);

            // INSERT CODE:
            // * Iterate over all neighbors and create a weighted average
            // * You can obtain the weight for an edge e by querying edgeWeight[e]
            //--- start strip ---





            //--- end strip ---

            // Set a fixed lambda (damping factor) of 0.5 and update the vertex position
            newPos[vh] = mesh.point(vh) + 0.5 * updateVector;
        }
    }


    // set new positions
    for (auto vh : mesh.vertices())
        if (!mesh.status(vh).locked())
            mesh.set_point(vh, newPos[vh]);

    emit updatedObject(_meshObject->id(), UPDATE_ALL);
}

//-----------------------------------------------------------------------------

void SummerSchoolSmootherPlugin::calcWeights(TriMeshObject* _meshObject)
{
    TriMesh& mesh = *_meshObject->mesh();
    auto edgeWeight
        = OpenMesh::PropertyManager<OpenMesh::EPropHandleT<double>, TriMesh>::createIfNotExists(mesh, "edgeWeight");

    // Uniform weighting
    if (weighting_ == UNIFORM)
    {
        for (auto eh : mesh.edges())
            edgeWeight[eh] = 1.0;
    }

    // Cotangent weighting
    else if (weighting_ == COTANGENT)
    {
        for (auto eh : mesh.edges())
        {
            double w = 0.0;

            TriMesh::HalfedgeHandle h0 = mesh.halfedge_handle(eh, 0);
            std::cout << "cotangent" << std::endl;
            TriMesh::VertexHandle v0 = mesh.to_vertex_handle(h0);
            TriMesh::Point p0 = mesh.point(v0);

            TriMesh::HalfedgeHandle h1 = mesh.halfedge_handle(eh, 1);
            TriMesh::VertexHandle v1 = mesh.to_vertex_handle(h1);
            TriMesh::Point p1 = mesh.point(v1);

            TriMesh::HalfedgeHandle h2 = mesh.next_halfedge_handle(h0);
            TriMesh::VertexHandle v2 = mesh.to_vertex_handle(h2);
            TriMesh::Point p2 = mesh.point(v2);

            TriMesh::Point d0 = (p0 - p2).normalize();
            TriMesh::Point d1 = (p1 - p2).normalize();
            w += 1.0 / tan(acos(d0 | d1));

            h2 = mesh.next_halfedge_handle(h1);
            p2 = mesh.point(mesh.to_vertex_handle(h2));
            d0 = (p0 - p2).normalize();
            d1 = (p1 - p2).normalize();
            w += 1.0 / tan(acos(d0 | d1));

            edgeWeight[eh] = w;
        }
    }

    emit updatedObject(_meshObject->id(), UPDATE_ALL);
}

//-----------------------------------------------------------------------------

void SummerSchoolSmootherPlugin::initializePlugin()
{
    gui_ = new SummerSchoolSmootherToolboxWidget();
    QSize size(100, 100);
    gui_->resize(size);

    connect(gui_->comboBox_weighting, SIGNAL(currentIndexChanged(int)), this, SLOT(slotWeightingChanged(int)));
    connect(gui_->pushButton_smooth, SIGNAL(clicked()), this, SLOT(slotSmooth()));

    emit addToolbox(tr("SummerSchoolSmoother"), gui_);
}

//-----------------------------------------------------------------------------

void SummerSchoolSmootherPlugin::slotWeightingChanged(int _idx)
{
    int type = gui_->comboBox_weighting->currentIndex();

    if (type == 0)
        weighting_ = UNIFORM;
    else if (type == 1)
        weighting_ = COTANGENT;

    // update: compute uniform or cotangent weights
    for (PluginFunctions::ObjectIterator o_it(PluginFunctions::ALL_OBJECTS, DataType(DATA_TRIANGLE_MESH));
         o_it != PluginFunctions::objectsEnd();
         ++o_it)
    {
        TriMeshObject* meshObject = PluginFunctions::triMeshObject(o_it);
        initMesh(meshObject);
        calcWeights(meshObject);
    }
}

//-----------------------------------------------------------------------------

void SummerSchoolSmootherPlugin::slotSmooth()
{
    for (PluginFunctions::ObjectIterator o_it(PluginFunctions::ALL_OBJECTS, DataType(DATA_TRIANGLE_MESH));
         o_it != PluginFunctions::objectsEnd();
         ++o_it)
    {
        TriMeshObject* meshObject = PluginFunctions::triMeshObject(o_it);
        initMesh(meshObject);
        smooth(meshObject);
    }
}

//-----------------------------------------------------------------------------

void SummerSchoolSmootherPlugin::initMesh(TriMeshObject* _meshObject)
{
    if (_meshObject == NULL)
        return;

    calcWeights(_meshObject);

    emit updatedObject(_meshObject->id(), UPDATE_ALL);
}

//-----------------------------------------------------------------------------

void SummerSchoolSmootherPlugin::smooth(TriMeshObject* _meshObject)
{
    int iters = gui_->spinBox_iterations->value();

    // lock one or two rings of boundary vertices
    lockBoundary(_meshObject);

    // smooth: compute and set new positions
    for (int i = 0; i < iters; ++i)
        computeNewPositions(_meshObject);

    // update normals
    _meshObject->mesh()->update_face_normals();
    _meshObject->mesh()->update_vertex_normals();

    emit updatedObject(_meshObject->id(), UPDATE_ALL);
}

//-----------------------------------------------------------------------------

void SummerSchoolSmootherPlugin::lockBoundary(TriMeshObject* _meshObject)
{
    TriMesh* mesh = _meshObject->mesh();


    // lock boundary vertices
    for (TriMesh::VertexIter v_it = mesh->vertices_begin(); v_it != mesh->vertices_end(); ++v_it)
        mesh->status(*v_it).set_locked(mesh->is_boundary(*v_it));


    emit updatedObject(_meshObject->id(), UPDATE_ALL);
}

//-----------------------------------------------------------------------------

#if QT_VERSION < 0x050000
Q_EXPORT_PLUGIN2(SummerSchoolSmootherPlugin, SummerSchoolSmootherPlugin);
#endif
