
/*
 * Types
 *
 * Author:  Henrik Zimmer <henrik@zimmer.to>
 *
 * Version: $Revision: 1 $
 * Date:    $Date: 2010-01-06 20:24:14 +0100 (Wed, 06 Jan 2010) $
 */

#ifndef HZ_TYPES_HH
#define HZ_TYPES_HH

#include <vector>
#include "Helpers.hh"
#include <OpenMesh/Core/IO/MeshIO.hh>
#include <OpenMesh/Core/Mesh/TriMesh_ArrayKernelT.hh>
#include <ObjectTypes/TriangleMesh/TriangleMesh.hh>
#include <ObjectTypes/PolyMesh/PolyMesh.hh>
#include <ACG/Math/VectorT.hh>
#include <Eigen/Eigen>
#include <Eigen/Sparse>
#include <ACG/Math/QuaternionT.hh>
#include <ACG/Math/Matrix4x4T.hh>
#include <ACG/Math/GLMatrixT.hh>
#include <ACG/Geometry/bsp/TriangleBSPT.hh> 

#include <ACG/Math/VectorT.hh>
#include <ACG/Utils/StopWatch.hh>
#include <ACG/Utils/ColorCoder.hh>

#include <ACG/Geometry/bsp/TriangleBSPT.hh> 
#include <ObjectTypes/TriangleMesh/TriangleMesh.hh>
#include <ObjectTypes/PolyMesh/PolyMesh.hh>

#include <bitset>
#include <istream>
#include <ostream>

#include <boost/unordered_map.hpp>

enum EdgeType { RED = 0, BLUE = 1, YELLOW = 2 };
enum LengthType { SHORTLEN = 0, MEDIUMLEN = 1, LONGLEN = 2 };

enum WeighType { CLOSENESS = 0, ORIENTATION = 1 };


const double golden( (1. +sqrt(5.))*0.5);
// Points, Scalars etc
typedef PolyMesh::Point::value_type Scalar;
typedef ACG::VectorT<Scalar, 3>     Vec3;
typedef ACG::VectorT<int, 3>        Vec3i;
typedef ACG::VectorT<Scalar, 2>     Vec2;
typedef ACG::VectorT<int, 2>     Vec2i;
typedef ACG::VectorT<int, 6>     Vec6i;
//typedef ACG::VectorT<int8_t, 6>     Vec6c;
typedef Vec6i Zome6dIntPos;

struct VecHash
{
  inline size_t operator()(const OpenMesh::VectorT<Scalar,3> & v) const
  {
    size_t seed = 0;
    boost::hash_combine(seed, v[0]);
    boost::hash_combine(seed, v[1]);
    boost::hash_combine(seed, v[2]);
    return seed;
  }
  inline size_t operator()(const OpenMesh::VectorT<int,6> & v) const
  {
    size_t seed = 0;
    boost::hash_combine(seed, v[0]);
    boost::hash_combine(seed, v[1]);
    boost::hash_combine(seed, v[2]);
    boost::hash_combine(seed, v[3]);
    boost::hash_combine(seed, v[4]);
    boost::hash_combine(seed, v[5]);
    return seed;
  }
};

struct ZomeCoordX2G
{
  public:

    ZomeCoordX2G( int _a=0, int _b=0): a_(_a), b_(_b) {}

    ZomeCoordX2G( Scalar _v)
    {
      if( fabs(fmod(_v, golden)) < 1e-6) // no b
      {
        if( _v >= 0.)
          a_ = (int)(_v / golden + 0.5);
        else
          a_ = (int)(_v / golden - 0.5);
        b_ = 0;
      }
      else // b != 0
      {
        double minintdiff(FLT_MAX);
        int bestb(0);
        int besta(0);
        for( int tmpa = -100; tmpa < 100; ++tmpa)
        {
          double curval(_v - golden * tmpa);
          double closestint(0.);
          if( curval >= 0.)
            closestint = (int)(curval + 0.5);
          else
            closestint = (int)(curval - 0.5);

          double intdiff( closestint - curval);
          if( fabs(intdiff) < minintdiff)
          {
            minintdiff = fabs(intdiff);
            besta = tmpa;
            bestb = closestint;
          }
        }
        a_ = besta;
        b_ = bestb;
      }
    }

    bool operator==( const ZomeCoordX2G& _rhs) const
    {
      return (a_ == _rhs.a_ && b_ == _rhs.b_);
    }

    ZomeCoordX2G& operator=( const ZomeCoordX2G& _rhs)
    {
      a_ = _rhs.a_;
      b_ = _rhs.b_;
      return *this;
    }

    bool operator!=( const ZomeCoordX2G& _rhs) const
    {
      return !(_rhs == *this);
    }

    ZomeCoordX2G operator*( const ZomeCoordX2G& _rhs) const
    {
      const int& c(_rhs.a_);
      const int& d(_rhs.b_);
      return ZomeCoordX2G( a_*c + a_*d + b_*c, a_*c + b_*d);
    }

    ZomeCoordX2G operator*( int _mult) const
    {
      return ZomeCoordX2G( a_*_mult, b_*_mult);
    }

    ZomeCoordX2G operator-( const ZomeCoordX2G& _rhs) const
    {
      const int& c(_rhs.a_);
      const int& d(_rhs.b_);
      return ZomeCoordX2G( a_-c, b_-d);
    }

    ZomeCoordX2G operator+( const ZomeCoordX2G& _rhs) const
    {
      const int& c(_rhs.a_);
      const int& d(_rhs.b_);
      return ZomeCoordX2G( a_+c, b_+d);
    }

    Vec2i operator()() const
    {
      return Vec2i( a_, b_);
    }

    Scalar scalar() const { return a_*golden + b_; }

    const int& a() const { return a_;};
          int& a()       { return a_;};
    const int& b() const { return b_;};
          int& b()       { return b_;};


  private:
  int a_; // factor of \gamma (golden ratio)
  int b_; // integer
};

struct ZomeCoordZT
{
  public:

    ZomeCoordZT( Vec3 _vec)
    {
      c[0] = ZomeCoordX2G(_vec[0]);
      c[1] = ZomeCoordX2G(_vec[1]);
      c[2] = ZomeCoordX2G(_vec[2]);
    }

    ZomeCoordZT( double _x = 0., double _y = 0., double _z = 0.)
    {
      c[0] = ZomeCoordX2G(_x);
      c[1] = ZomeCoordX2G(_y);
      c[2] = ZomeCoordX2G(_z);
    }
    ZomeCoordZT( ZomeCoordX2G _x, ZomeCoordX2G _y, ZomeCoordX2G _z)
    {
      c[0] = _x;
      c[1] = _y;
      c[2] = _z;
    }
    ZomeCoordZT( const Vec6i& _vec6i)
    {
      c[0].a() = _vec6i[0];
      c[0].b() = _vec6i[1];
      c[1].a() = _vec6i[2];
      c[1].b() = _vec6i[3];
      c[2].a() = _vec6i[4];
      c[2].b() = _vec6i[5];
    }

    //bool operator==( const ZomeCoordZT& _rhs) const
    //{
    //  Vec6i rhs(0,0,0,0,0,0);
    //  Vec6i cur(0,0,0,0,0,0);
    //  _rhs(rhs);
    //  (*this)(cur);

    //  return (rhs == cur);
    //}

    ZomeCoordZT mult_golden( ) const 
    {
      ZomeCoordX2G go(1,0);
      ZomeCoordZT ans;
      ans.c[0] = c[0]*go;
      ans.c[1] = c[1]*go;
      ans.c[2] = c[2]*go;
      return ans;
    }

    bool operator==( const ZomeCoordZT& _rhs) const
    {
      if( (c[0] == _rhs.c[0]) && 
          (c[1] == _rhs.c[1]) && 
          (c[2] == _rhs.c[2]))
        return true;
      return false;
    }
    bool operator!=( const ZomeCoordZT& _rhs) const
    {
      return !(_rhs == *this);
    }

    ZomeCoordZT& operator=( const ZomeCoordZT& _rhs) 
    {
      c[0] = _rhs.c[0];
      c[1] = _rhs.c[1];
      c[2] = _rhs.c[2];
      return *this;
    }
    ZomeCoordZT operator+( const ZomeCoordZT&  _rhs) const
    {
      return ZomeCoordZT( _rhs.c[0]+c[0], _rhs.c[1]+c[1], _rhs.c[2]+c[2]);
    }
    ZomeCoordZT operator-( const ZomeCoordZT&  _rhs) const
    {
      return ZomeCoordZT( c[0] - _rhs.c[0], c[1] - _rhs.c[1], c[2] - _rhs.c[2]);
    }
    ZomeCoordZT operator*( int  _mult) const
    {
      return ZomeCoordZT( c[0]*_mult, c[1]*_mult, c[2]*_mult);
    }

  Vec3 vec3() const
  {
    return Vec3( c[0].scalar(), c[1].scalar(), c[2].scalar());
  }
  operator Vec3() const
  {
    return Vec3( c[0].scalar(), c[1].scalar(), c[2].scalar());
  }

  Vec6i vec6i() const
  {
    Vec6i vec;
    vec[0] = c[0].a();
    vec[1] = c[0].b();
    vec[2] = c[1].a();
    vec[3] = c[1].b();
    vec[4] = c[2].a();
    vec[5] = c[2].b();
    return vec;
  }

  operator Vec6i() const
  {
    return Vec6i( c[0].a(), c[0].b(), c[1].a(), c[1].b(), c[2].a(), c[2].b());
  }

  Vec3 operator()() const
  {
    return Vec3( c[0].scalar(), c[1].scalar(), c[2].scalar());
  }

  Vec6i operator()( Vec6i& _vec) const
  {
    _vec[0] = c[0].a();
    _vec[1] = c[0].b();
    _vec[2] = c[1].a();
    _vec[3] = c[1].b();
    _vec[4] = c[2].a();
    _vec[5] = c[2].b();
    return _vec;
  }

  private:
  ZomeCoordX2G c[3];
};


namespace ACG
{
  struct ZomePlane
  {
    ZomePlane() : orthogonalholeid(-1) {}
    int orthogonalholeid;

    Vec3 normal;
    std::vector< int > holes;
  };

}

typedef OpenMeshTriangleBSPT<TriMesh> BSPT;

// Meshes etc
typedef PolyMesh MeshT;
typedef MeshT::FaceIter FaceIter;
typedef MeshT::FaceIter FIter;
typedef MeshT::EdgeIter EdgeIter;
typedef MeshT::EdgeIter EIter;
typedef MeshT::VertexIter VertexIter;
typedef MeshT::VertexIter VIter;
typedef MeshT::HalfedgeIter HalfedgeIter;
typedef MeshT::FaceVertexIter FVIter;
typedef MeshT::ConstFaceVertexIter CFVIter;
typedef MeshT::FaceEdgeIter   FEIter;
typedef MeshT::HalfedgeIter HIter;
typedef MeshT::FaceHalfedgeIter FHIter;
typedef MeshT::VertexVertexIter VVIter;
typedef MeshT::VertexEdgeIter VEIter;
typedef MeshT::ConstVertexEdgeIter CVEIter;
typedef MeshT::FaceEdgeIter FEIter;
typedef MeshT::ConstFaceEdgeIter CFEIter;
typedef MeshT::VertexOHalfedgeIter VOHIter;
typedef MeshT::CVOHIter            CVOHIter;
typedef MeshT::FaceFaceIter     FFIter;
typedef MeshT::VertexFaceIter VFIter;
typedef MeshT::ConstVertexFaceIter CVFIter;
typedef MeshT::HalfedgeHandle HEH;
typedef MeshT::FaceHandle FH;
typedef MeshT::FaceHandle FH;
typedef MeshT::VertexHandle VH;
typedef MeshT::EdgeHandle EH;
typedef MeshT::Color Color;

// Eigen Matrices etc
typedef Eigen::VectorXd VectorXd;
typedef Eigen::MatrixXd MatrixXd;
typedef Eigen::Matrix4d Matrix4d;
typedef Eigen::Triplet<Scalar> Triplet;
typedef Eigen::SparseMatrix< Scalar > SpMatrix;
typedef ACG::Matrix4x4T<Scalar> Matrix4x4;
typedef ACG::QuaternionT<Scalar> Quaternion;

// first 8 bits are permutation, following 1 bit is reversed flag
struct PermT
{
  PermT() {perm_ = 0;}
  int perm() const { return (perm_ & (0xFF)); }
  int reverse() const { return (perm_ & 0x100) >> 8; }
  void set_perm( int _perm) { perm_ = ((perm_ & ~(0xFF)) | (_perm & 0xFF)); }
  void set_reverse( bool _rev) { perm_ = (perm_ & ~(0x100)); if( _rev) perm_ |= 0x100; }
  int perm_;
};

// number generator functor for e.g. vectors
struct IntEnumeratorGenerator {
  int current;
  IntEnumeratorGenerator() {current=0;}
  int operator()() {return current++;}
};

#endif 
