#include "ZomeDirections.hh"

namespace ACG
{

const std::vector< Vec3 >& ZomeDirections::dirs3d() const
{
  return dirs3d_;
}

const std::vector< ZomeCoordZT >& ZomeDirections::dirs6d() const
{
  return dirs6d_;
}

const Vec3& ZomeDirections::dir3d( int _id) const
{
  return dirs3d_[_id];
}

const ZomeCoordZT& ZomeDirections::dir6dZT( int _id) const
{
  return dirs6d_[_id];
}

Zome6dIntPos ZomeDirections::dir6dvec6i( int _id) const
{
  return dirs6d_[_id].vec6i();
}

size_t ZomeDirections::id( const OneStrutConfiguration& _hl) const
{
  return _hl.get_h() + 62 * _hl.get_l();
}

size_t ZomeDirections::id( int _h, int _l) const
{
  return _h + 62 * _l;
}

size_t ZomeDirections::oppid( size_t _i) const
{
  return oppid_[_i];
}

size_t ZomeDirections::n_dirs( ) const
{
  return dirs3d_.size();
}

void ZomeDirections::setup_base_directions( )
{
  dirs3d_.reserve(62*3);
  dirs3d_.clear();
  
  //blues
  //0-5 -- axis aligned
  dirs3d_.push_back( Vec3(  1., 0., 0.));
  dirs3d_.push_back( Vec3( -1., 0., 0.));
  dirs3d_.push_back( Vec3( 0.,  1., 0.));
  dirs3d_.push_back( Vec3( 0., -1., 0.));
  dirs3d_.push_back( Vec3( 0., 0.,  1.));
  dirs3d_.push_back( Vec3( 0., 0., -1.));

  // diagonals
  // around x axis (6-13)
  dirs3d_.push_back( Vec3( 0.5*golden, 0.5/golden, 0.5));
  dirs3d_.push_back( Vec3( 0.5*golden, -0.5/golden, 0.5));
  dirs3d_.push_back( Vec3( -0.5*golden, 0.5/golden, 0.5));
  dirs3d_.push_back( Vec3( -0.5*golden, -0.5/golden, 0.5));

  dirs3d_.push_back( Vec3( -0.5*golden, -0.5/golden, -0.5));
  dirs3d_.push_back( Vec3( -0.5*golden, 0.5/golden, -0.5));
  dirs3d_.push_back( Vec3( 0.5*golden, -0.5/golden, -0.5));
  dirs3d_.push_back( Vec3( 0.5*golden, 0.5/golden, -0.5));

  // around y axis (14-21)
  dirs3d_.push_back( Vec3( 0.5, 0.5*golden, 0.5/golden));
  dirs3d_.push_back( Vec3( -0.5, 0.5*golden, 0.5/golden));
  dirs3d_.push_back( Vec3( 0.5, -0.5*golden, 0.5/golden));
  dirs3d_.push_back( Vec3( -0.5, -0.5*golden, 0.5/golden));

  dirs3d_.push_back( Vec3( -0.5, -0.5*golden, -0.5/golden));
  dirs3d_.push_back( Vec3( 0.5, -0.5*golden, -0.5/golden));
  dirs3d_.push_back( Vec3( -0.5, 0.5*golden, -0.5/golden));
  dirs3d_.push_back( Vec3( 0.5, 0.5*golden, -0.5/golden));

  // around z axis (22-29)
  dirs3d_.push_back( Vec3( 0.5/golden, 0.5, 0.5*golden));
  dirs3d_.push_back( Vec3( -0.5/golden, 0.5, 0.5*golden));
  dirs3d_.push_back( Vec3( 0.5/golden, -0.5, 0.5*golden));
  dirs3d_.push_back( Vec3( -0.5/golden, -0.5, 0.5*golden));

  dirs3d_.push_back( Vec3( -0.5/golden, -0.5, -0.5*golden));
  dirs3d_.push_back( Vec3( 0.5/golden, -0.5, -0.5*golden));
  dirs3d_.push_back( Vec3( -0.5/golden, 0.5, -0.5*golden));
  dirs3d_.push_back( Vec3( 0.5/golden, 0.5, -0.5*golden));

  // reds (diagonals in planes)
  // around x axis (30-33)
  dirs3d_.push_back( Vec3(  golden*0.5, 0.5, 0.));
  dirs3d_.push_back( Vec3(  golden*0.5, -0.5, 0.));
  dirs3d_.push_back( Vec3(  -golden*0.5, -0.5, 0.));
  dirs3d_.push_back( Vec3(  -golden*0.5, 0.5, 0.));
  
  // around y axis (34-37)
  dirs3d_.push_back( Vec3(  0., golden*0.5, 0.5));
  dirs3d_.push_back( Vec3(  0., golden*0.5, -0.5));
  dirs3d_.push_back( Vec3(  0., -golden*0.5, -0.5));
  dirs3d_.push_back( Vec3(  0., -golden*0.5, 0.5));

  // around z axis (38-41)
  dirs3d_.push_back( Vec3( 0.5, 0., golden*0.5));
  dirs3d_.push_back( Vec3( -0.5, 0., golden*0.5));
  dirs3d_.push_back( Vec3( -0.5, 0., -golden*0.5));
  dirs3d_.push_back( Vec3( 0.5, 0., -golden*0.5));

  // yellows (diagonals in planes)
  // around x axis (42-35)
  dirs3d_.push_back( Vec3( golden*0.5, 0.,  0.5/golden));
  dirs3d_.push_back( Vec3( golden*0.5, 0., -0.5/golden));
  dirs3d_.push_back( Vec3( -golden*0.5, 0., -0.5/golden));
  dirs3d_.push_back( Vec3( -golden*0.5, 0., 0.5/golden));

  // around y axis (46-49)
  dirs3d_.push_back( Vec3( 0.5/golden, 0.5*golden, 0.));
  dirs3d_.push_back( Vec3( -0.5/golden, 0.5*golden, 0.));
  dirs3d_.push_back( Vec3( -0.5/golden, -0.5*golden, 0.));
  dirs3d_.push_back( Vec3( 0.5/golden, -0.5*golden, 0.));

  // around z axis (50-53)
  dirs3d_.push_back( Vec3( 0., 0.5/golden, golden*0.5));
  dirs3d_.push_back( Vec3( 0., -0.5/golden, golden*0.5));
  dirs3d_.push_back( Vec3( 0., -0.5/golden, -golden*0.5));
  dirs3d_.push_back( Vec3( 0., 0.5/golden, -golden*0.5));

  // 54-61 (diagonals)
  dirs3d_.push_back( Vec3( 0.5, 0.5, 0.5));
  dirs3d_.push_back( Vec3( -0.5, 0.5, 0.5));
  dirs3d_.push_back( Vec3( 0.5, -0.5, 0.5));
  dirs3d_.push_back( Vec3( 0.5, 0.5, -0.5));
  dirs3d_.push_back( Vec3( 0.5, -0.5, -0.5));
  dirs3d_.push_back( Vec3( -0.5, 0.5, -0.5));
  dirs3d_.push_back( Vec3( -0.5, -0.5, 0.5));
  dirs3d_.push_back( Vec3( -0.5, -0.5, -0.5));

  if( !test_base_directions())
    std::cerr << __FUNCTION__ << "\n\n\n\t\t\t ERROR Something wrong with the base directions \n\n\n" << std::endl;
}

bool ZomeDirections::test_base_directions( ) 
{
  bool allok(true);
  // 1) test direction norm
  Scalar b0norm( 1.);
  Scalar y0norm( sqrt(3.)/2.);
  Scalar r0norm( sqrt(2.+golden)/2.);
  for( size_t i = 0; i < 62; ++i)
  {
    Scalar currnorm( dirs3d_[i].length());
    if( !( (fabs(currnorm-b0norm) < 1e-6) || (fabs(currnorm-y0norm) < 1e-6) || (fabs(currnorm-r0norm) < 1e-6)))
    {
      allok = false;
      std::cerr << __FUNCTION__ << " Warning bad norm! " << i << " ... " << currnorm << std::endl;
    }
  }

  // 3) test no two the same
  for( size_t i = 0; i < dirs3d_.size(); ++i)
  {
    for( size_t j = 0; j < dirs3d_.size(); ++j)
    {
      if( j!= i)
      {
        if( dirs3d_[i] == dirs3d_[j])
        {
          allok = false;
          std::cerr << " dir " << i << " and " << j << " are the same!" << std::endl;
        }
      }
    }
  }

  // 2) test against other (global_node_data) directions
  map_id_to_old_h_l_.clear();
  map_id_to_old_h_l_.resize( 3*62);
  for( size_t i = 0; i < dirs3d_.size(); ++i)
  {
    Vec3 testdir1( dirs3d_[i]);
    testdir1.normalize();
    Vec3 bestdir( 0,0,0);
    Scalar besterror(FLT_MAX);
    int besth(-1);

    for( size_t h = 0; h < 62; ++h)
    {
      Vec3 tmpdir( global_node_data().dir(h));
      tmpdir.normalize();
      Vec3 diff( tmpdir - testdir1);
      Scalar err( diff.norm());
      if( err < besterror)
      {
        besterror = err;
        bestdir = tmpdir;
        besth = h;
      }
    }
    if( besterror > 1e-6)
    {
      std::cerr << __FUNCTION__ << " Warning, no match for direction " << i << " against working directions... error " << besterror << std::endl;
      allok = false;
    }
    else
    {
      map_id_to_old_h_l_[i+0] = std::make_pair( besth, 0);
      map_id_to_old_h_l_[i+62] = std::make_pair( besth, 1);
      map_id_to_old_h_l_[i+124] = std::make_pair( besth, 2);
    }
  }
  return allok;
}

void ZomeDirections::triple_base_directions( )
{
  map_id_to_base_range_.clear();
  map_id_to_base_range_.resize(62*3);
  for( size_t i = 0; i < 62; ++i)
    map_id_to_base_range_[i] = i;

  for( size_t f = 0; f < 2; ++f)
    for( size_t i = f*62; i < (f+1)*62; ++i)
    {
      dirs3d_.push_back( dirs3d_[i] * golden);
      map_id_to_base_range_[i+62] = i-f*62;
    }
}

void ZomeDirections::setup_opposite_directions( )
{
  //std::cerr << __FUNCTION__ << " number of directions to start with " <<  dirs3d_.size() << std::endl;
  oppid_.clear();
  oppid_.resize( dirs3d_.size(), 0);
  for( size_t i = 0; i < dirs3d_.size(); ++i)
  {
    Vec3 diri( -dirs3d_[i]);
    Scalar besterror(FLT_MAX);
    size_t bestj( 0);
    for( size_t j = 0; j < dirs3d_.size(); ++j)
    {
      Vec3 dirj( dirs3d_[j]);
      Scalar currerror( (diri-dirj).norm());
      if( currerror < besterror)
      {
        besterror = currerror;
        bestj = j;
      }
    }
    if( besterror > 1e-6)
      std::cerr << __FUNCTION__ << " WARNING ERROR must find the same direction ... " << i << " vs " << bestj << " besterror " << besterror << std::endl;
    oppid_[i] = bestj;
    //std::cerr << i << " <-> " << bestj << " base " << map_id_to_base_range_[i] << std::endl;
  }
}

void ZomeDirections::map_id_to_old_h_l( int _i, int& _h, int& _l) const
{
  _h = map_id_to_old_h_l_[_i].first;
  _l = map_id_to_old_h_l_[_i].second;
}

int ZomeDirections::map_id_to_old_h( int _i) const
{
  return map_id_to_old_h_l_[_i].first;
}

size_t ZomeDirections::map_to_base_range( size_t _i) const
{
  return map_id_to_base_range_[_i];
}

bool ZomeDirections::diridfrom6d( const Zome6dIntPos& _pos, size_t& _dirid) const
{
  boost::unordered_map< Vec6i, size_t, VecHash >::const_iterator mapit = map6dtodirid_.find( _pos);
  if( mapit != map6dtodirid_.end())
  {
    _dirid = mapit->second;
    return true;
  }
  else
  {
    _dirid = std::numeric_limits<size_t>::max();
    return false;
  }
}

void ZomeDirections::setup_2gammascaled_directions( )
{
  dirs6d_.reserve(62*3);
  dirs6d_.clear();

  // first 62 base directions
  for( size_t i = 0; i < 62; ++i)
    dirs6d_.push_back( ZomeCoordZT( dirs3d_[i] * 2. * golden));

  // convert the others in integer space
  for( size_t f = 0; f < 2; ++f)
  {
    for( size_t i = f*62; i < (f+1)*62; ++i)
    {
      dirs6d_.push_back( dirs6d_[i].mult_golden());
    }
  }

  for( size_t i = 0; i < dirs6d_.size(); ++i)
    map6dtodirid_[dirs6d_[i]] = i;

  if( !test_2gammascaled_directions())
    std::cerr << __FUNCTION__ << "\n\n\n\t\t\t ERROR Something wrong with the integer (2*gamma) directions \n\n\n" << std::endl;
}

bool ZomeDirections::test_2gammascaled_directions( ) const
{
  bool allok = true;
  // convert back to 3d and compare to originals
  for( size_t i = 0; i < dirs3d_.size(); ++i)
  {
    Vec3 diff( dirs3d_[i] * 2. * golden - dirs6d_[i].vec3());
    if( diff.norm() > 1e-6)
    {
      std::cerr << __FUNCTION__ << " Conversion error !! " << dirs3d_[i]*2.*golden << " vs " << dirs6d_[i].vec3() << std::endl;
      allok = false;
    }
  }

  // 2) are two the same?
  for( size_t i = 0; i < dirs6d_.size(); ++i)
  {
    for( size_t j = 0; j < dirs6d_.size(); ++j)
    {
      if( j!=i)
      {
        if( dirs6d_[i] == dirs6d_[j])
        {
          std::cerr << __FUNCTION__ << " i and j are the same " << i << " vs " << j << std::endl;
          allok = false;
        }
      }
    }
  }
  // 3) same opposites?
  for( size_t i = 0; i < dirs6d_.size(); ++i)
  {
    for( size_t j = 0; j < dirs6d_.size(); ++j)
    {
      if( Vec6i(dirs6d_[i]) == -Vec6i(dirs6d_[j]))
      {
        if( oppid(i) != j || oppid(j) != i)
        {
          std::cerr << __FUNCTION__ << " opposites do not match!! " << i << " vs " << j << " and " << oppid(i) << " vs " << oppid(j) << std::endl;
          allok = false;
        }
      }
    }
  }
  return allok;
}


void ZomeDirections::test_all_directions( ) const
{
  Scalar scalefactor[3];
  scalefactor[BLUE] = 1.;
  scalefactor[RED] = sqrt(2.+golden) * 0.5;
  scalefactor[YELLOW] = sqrt(3.) * 0.5;


  Vec3 xaxis( dirs3d_[0]);
  // find old xaxis
  bool done(false);
  for( size_t i = 0; i < 62 && !done; ++i)
  {
    if( (global_node_data().dir( i).normalize() - xaxis).norm() < 1e-6 )
    {
      done = true;
      //std::cerr << " found xaxis " << global_node_data.dir(i) << " vs " << xaxis << std::endl;
    }
  }

  // find all 
  std::vector< std::pair< int, int > > mapping;
  for( size_t i = 0; i < 62*3; ++i)
  {
    Vec3 currentdir( dirs3d_[i]);
    //std::cerr << " currentdir " << currentdir<< " ";
    done = false;
    int besth(-1);
    int bestl(-1);
    Vec3 bestdir(0,0,0);
    for( int h = 0; h < 62 && !done; ++h)
      for( int l = 0; l < 3; ++l)
      {
        Scalar factor( scalefactor[global_node_data().type(h)]);
        //std::cerr << " type " << global_node_data.type(h) << " factor " << factor << std::endl;
        if( l > 0)
          factor *= golden;
        if( l > 1)
          factor *= golden;
        Vec3 thisdir( global_node_data().dir(h)*factor);
        if( (currentdir - thisdir).norm() < 1e-6)
        {
          done = true;
          besth = h;
          bestdir = thisdir;
          bestl = l;
        }
      }
    if( done)
    {
     // std::cerr << " found h " << besth << " l " << bestl << " dir " << bestdir << std::endl;
      mapping.push_back( std::make_pair(besth, bestl));
    }
    else
      std::cerr << " BAD BAD !! " << std::endl;
  }

  //std::cerr << " CHECKING 6D DIRS! " << std::endl;
  for( size_t i = 0; i < 62*3; ++i)
  {
    Vec6i currentdir6( dirs6d_[i]);
    size_t mappedid(0);
    diridfrom6d( currentdir6, mappedid);
    if( mappedid != i )
      std::cerr << "\t BAD BAD BAD mappedid " << mappedid << " vs " << i << std::endl;

    Vec3 currentdir( dirs6d_[i]);
    currentdir /= 2.*golden;
    //std::cerr << " currentdir " << currentdir6<< " and " << currentdir ;
    done = false;
    int besth(-1);
    int bestl(-1);
    Vec3 bestdir(0,0,0);
    for( int h = 0; h < 62 && !done; ++h)
      for( int l = 0; l < 3; ++l)
      {
        Scalar factor( scalefactor[global_node_data().type(h)]);
        //std::cerr << " type " << global_node_data.type(h) << " factor " << factor << std::endl;
        if( l > 0)
          factor *= golden;
        if( l > 1)
          factor *= golden;
        Vec3 thisdir( global_node_data().dir(h)*factor);
        if( (currentdir - thisdir).norm() < 1e-6)
        {
          done = true;
          besth = h;
          bestdir = thisdir;
          bestl = l;
        }
      }
    if( done)
    {
     // std::cerr << " found h " << besth << " l " << bestl << " dir " << bestdir << std::endl;
      if( mapping[i].first != besth || mapping[i].second != bestl )
      {
        std::cerr << " WTF mapping error " << besth << " and " << bestl << " vs " << mapping[i].first << " and " << mapping[i].second << std::endl;
      }
    }
    else
      std::cerr << " BAD BAD !! " << std::endl;
  }

}

}

