//=============================================================================
//
//  CLASS ZomeMeshViewerPlugin - IMPLEMENTATION
//
//=============================================================================

//== INCLUDES =================================================================

#include <Qt>
#include <QtGui>
#include <QSpacerItem>

#include "ZomeMeshViewerPlugin.hh"

#include <iostream>

#include <OpenFlipper/BasePlugin/PluginFunctions.hh>

#include <OpenFlipper/BasePlugin/RPCWrappers.hh>

#undef min
#undef max


//== IMPLEMENTATION ==========================================================

ZomeMeshViewerPlugin::~ZomeMeshViewerPlugin() {};


ZomeMeshViewerPlugin::ZomeMeshViewerPlugin():tool_(0) {};


void ZomeMeshViewerPlugin::initializePlugin()
{

  tool_ = new ZomeMeshViewerToolbar();
  
  QSize size(300,300);
  tool_->resize(size);
  
  connect(tool_->load_pb, SIGNAL( clicked() ), this, SLOT( slot_load( ) ) );
  connect(tool_->draw_pb, SIGNAL( clicked() ), this, SLOT( slot_draw() ) );
  connect(tool_->drawnodes_cb, SIGNAL( stateChanged(int) ), this, SLOT( slot_draw() ) );

  connect(tool_->export_vzome_pb, SIGNAL( clicked() ), this, SLOT( slot_export_vzome() ) );

  emit addToolbox( tr("Zome Mesh Viewer") , tool_ );


 QString boostlicense = " ZomeMeshViewer uses the POCO libraries: www.pocoproject.org <br><br> The Boost Software License 1.0 <br> Permission is hereby granted, free of charge, to any person or organization obtaining a copy of the software and accompanying documentation covered by this license (the \"Software\") to use, reproduce, display, distribute, execute, and transmit the Software, and to prepare derivative works of the Software, and to permit third-parties to whom the Software is furnished to do so, all subject to the following: <br> The copyright notices in the Software and this entire statement, including the above license grant, this restriction and the following disclaimer, must be included in all copies of the Software, in whole or in part, and all derivative works of the Software, unless such copies or derivative works are solely in the form of machine-executable object code generated by a source language processor.  <br> THE SOFTWARE IS PROVIDED \"AS IS\", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.  ";

 emit addAboutInfo(boostlicense, "ZomeMeshViewer Plugin");
}


void ZomeMeshViewerPlugin::pluginsInitialized() 
{
}

////////////////////////////////////////////////////////

ZomeMeshViewerPlugin::POD*
ZomeMeshViewerPlugin::
get_POD( BaseObjectData* _object )
{
  // initialize PerObjectData if not done yet
  if (!_object->hasObjectData(pod_name()))
  {
    // get mesh object
    PolyMesh* mesh = dynamic_cast< PolyMeshObject* >( _object )->mesh();
    int pid( _object->id());

    // initialize per object data
    _object->setObjectData(pod_name(), new POD(*mesh, pid));
  }
  
  return (dynamic_cast< POD* >(_object->objectData(pod_name() )));
}

////////////////////////////////////////////////////////

ZomeMeshViewerPlugin::ZMM*
ZomeMeshViewerPlugin::
get_ZMM( BaseObjectData* _object )
{
  return dynamic_cast< ZMM *>(& get_POD( _object)->zmm());
}

////////////////////////////////////////////////////////

void ZomeMeshViewerPlugin::create_zomemesh( PolyMeshObject*& _polymeshobj, int& _polymeshid, PolyMesh*& _polymesh )
{
  // add PolyMesh for topology of ZomeMesh
  int tmpid;
  int polymeshobj_id;
  BaseObjectData *obj;

  emit addEmptyObject(DATA_POLY_MESH, tmpid);
  PluginFunctions::getObject(tmpid, obj);

  _polymeshobj = PluginFunctions::polyMeshObject(obj);
  _polymeshid = _polymeshobj->id();
  _polymeshobj->show();
  _polymeshobj->target(true);
  _polymeshobj->meshNode()->drawMode(ACG::SceneGraph::DrawModes::SOLID_FLAT_SHADED);
  _polymesh = _polymeshobj->mesh();


  // create POD object for topology mesh
  POD* pod( get_POD( _polymeshobj));
  pod->iszomemesh() = true;

  // add 4 PolyMeshes for struts and nodes (drawing)
  // these are stored in the POD structure (and thus connected to the topology mesh)
  std::string meshnames[4];
  meshnames[BLUE] = "Blue Struts";
  meshnames[RED] = "Red Struts";
  meshnames[YELLOW] = "Yellow Struts";
  meshnames[3] = "Nodes";
  for( int i = 0; i < 4; ++i)
  {
    // find zomemesh mesh
    PolyMeshObject  *polymeshobj(NULL);
    int& polymesh_id(pod->drawmeshes().id[i]);

    emit addEmptyObject(DATA_POLY_MESH, polymesh_id);
    PluginFunctions::getObject(polymesh_id, obj);
    polymeshobj = PluginFunctions::polyMeshObject(obj);
    polymeshobj->setName(meshnames[i].c_str());
    polymeshobj->show();
    polymeshobj->target(true);
    polymeshobj->meshNode()->drawMode(ACG::SceneGraph::DrawModes::SOLID_FACES_COLORED_FLAT_SHADED);
    pod->drawmeshes().meshp[i] = polymeshobj->mesh();
    pod->drawmeshes().meshp[i]->clean();
    Color col;
    if( (unsigned int)i == YELLOW)
    {
      col = Color(209./255., 167./255., 14./255., 1.);
    }
    else if( (unsigned int)i == BLUE)
    {
      col = Color( 13./255,88./255,138./255, 1.);
    }
    else if( (unsigned int)i == RED)
    {
      col = Color( 143./255., 34./255., 28./255., 1.);
    }
    else if( i == 3) // Node
      col = Color( 153./255.,153./255.,153./255.,1.0);
    else if( i == 4) // topology
    {
      col = Color( 67./255.,67./255.,67./255.,1.);
      dynamic_cast< PolyMeshObject* >( (obj) )->materialNode()->set_ambient_color( Color( 1., 1., 1., 1.0));
    }

    dynamic_cast< PolyMeshObject* >( (obj) )->materialNode()->set_base_color( Color(0.,0.,0.,1.));
    if( i != 4)
      dynamic_cast< PolyMeshObject* >( (obj) )->materialNode()->set_ambient_color( col);

    if( i != 3)
    {
      dynamic_cast< PolyMeshObject* >( (obj) )->materialNode()->set_diffuse_color( col);
    }
    dynamic_cast< PolyMeshObject* >( (obj) )->materialNode()->set_specular_color( Color(0.,0.,0.,1.));
  }
}


////////////////////////////////////////////////////////

void ZomeMeshViewerPlugin::slot_load( )
{

  QString fileName = QFileDialog::getOpenFileName(NULL, tr("Load ZomeMesh"),
      QDir::homePath() + "/data/ZomeMeshes/", tr("ZomeMeshes (*.zm2 *.zmc)"));

  if( fileName == "")
  {
    std::cerr << __FUNCTION__ << ":: No file selected, loading aborted!" << std::endl;
    return;
  }

  // add new PolyMesh (and setup POD and drawmeshes)
  PolyMesh* polymesh(NULL);
  int polymeshid(-1);
  PolyMeshObject* polymeshobj(NULL);
  create_zomemesh( polymeshobj, polymeshid, polymesh);

  // get POD
  POD* pod( get_POD( polymeshobj));

  // parse filename
  QFileInfo fi( fileName);


  std::string completefilewithoutending( (fi.absolutePath() + "/"+ fi.baseName()).toStdString());
  if( fi.suffix() == "zm2") // loading pair of .off and .zm2 files
  {
    pod->zmm().load( completefilewithoutending, 1);
  }
  else if( fi.suffix() == "zmc") // loading zip-container containing .off and .zm2 file (with same name)
  {
    pod->zmm().load( completefilewithoutending, 2);
  }
  else
  {
    std::cerr << __FUNCTION__ << ":: Unknown suffix " << fi.suffix().toStdString() << ", aborting loading!" << std::endl;
    return;
  }

  // set name of PolyMesh to name from file
  polymeshobj->setName("Topology");

  // add group collecting all meshes
  IdList objectIDs;

  objectIDs.push_back( polymeshid);
  for( int i = 0; i < 4; ++i)
    objectIDs.push_back(pod->drawmeshes().id[i]);

  QString gname(fi.baseName());
  gname.append(" group");
  RPC::callFunction ("datacontrol", "groupObjects", objectIDs, gname);

  // call draw function to fill color meshes and update normals etc.
  slot_draw( );
}

////////////////////////////////////////////////////////

void ZomeMeshViewerPlugin::slot_draw( )
{
  for ( PluginFunctions::ObjectIterator o_it(PluginFunctions::TARGET_OBJECTS,DATA_POLY_MESH) ; o_it != PluginFunctions::objectsEnd(); ++o_it) 
  {
    PolyMesh* mesh = dynamic_cast< PolyMeshObject* >( *o_it )->mesh();
    int meshid( (*o_it)->id());

    POD* pod( get_POD( *o_it));
    if( !pod->iszomemesh())
      continue;

    // color counters (RED, BLUE, YELLOW)
    std::vector< std::vector< int > > colorcounters(3, std::vector< int >(3, 0));
    std::vector< int > colorcounterssum(3, 0);

    
    ZMM& zmm( pod->zmm());
    ACG::ZomeDirections& dirs(pod->directions());

    Scalar blueglobalscale( zmm.scale()*golden);

    global_strut_data()[0].rescale( blueglobalscale/golden);
    global_strut_data()[1].rescale( blueglobalscale/golden);
    global_strut_data()[2].rescale( blueglobalscale/golden);
    
    // scale relating real-life zome nodes to the virtual ones
    Scalar spherediam( 0.19098*blueglobalscale);

    // transform zomemesh to "real" coordinates from 6D rep.
    zmm.transform_to_real( );

    // clear old meshes
    for( int i = 0; i < 4; ++i)
      pod->drawmeshes().meshp[i]->clean();

    // don't always draw nodes, since they are geometrically quite complex and too many of them can lead to memory issues
    if( tool_->drawnodes_cb->isChecked())
    {
      for( size_t i = 0; i < mesh->n_vertices(); ++i)
      {
        global_node_data().add_visual_mesh_to_mesh( *(pod->drawmeshes().meshp[3]), mesh->point( VH(i)), spherediam);
      }
    }

    for( size_t e = 0; e < mesh->n_edges(); ++e)
    {
      if( mesh->status(EH(e)).deleted())
        continue;

      HEH h0( mesh->halfedge_handle( EH(e),0));
      VH  v0( mesh->to_vertex_handle(h0));
      VH  v1( mesh->from_vertex_handle(h0));
      int h,l;
      dirs.map_id_to_old_h_l( zmm.dirid(h0), h, l);

      Vec3 startpos( zmm.transform_to_real( zmm.pos( v1)));

      EdgeType type( global_node_data().type(h));
      Vec3 dir( global_node_data().dir(h));
      Scalar len( global_strut_data()[type].len(l));

      ACG::ZomeStrutData* strutbase = &global_strut_data()[type];

      colorcounters[type][l]++; 
      colorcounterssum[type]++; 

      Vec3 origin, xaxis, zaxis;
      Scalar scale;
      Scalar balldim;
      global_node_data().get_strut_scale_origin_zaxis_and_xaxis( spherediam, h, balldim, scale, origin, zaxis, xaxis);
      origin += startpos;
      strutbase->get_strut_geometry( l, balldim, scale, origin, zaxis, xaxis, *(pod->drawmeshes().meshp[type]));
    }


    for( size_t i = 0; i < 4; ++i)
    {
      pod->drawmeshes().meshp[i]->update_normals();
      emit updatedObject( pod->drawmeshes().id[i], UPDATE_ALL);
    }

    // For now we know max. face valence = 4
    //std::vector< Vec3 > normals;
    //std::vector< Vec3 > points;

    //// Use a reasonable initial size
    //normals.resize(20);
    //points.resize(20);
    Vec3 points[4];
    Vec3 normals[4];

    // fix normals of topology mesh
    for( size_t i = 0; i < mesh->n_faces(); ++i)
    {
      const int val( mesh->valence( FH(i)));
	
      // Test once and resize both to avoid unnecessary lookups
      //if ( val > normals.size() ) {
      //  normals.resize(val);
      //  points.resize(val);
      //}

      int j(0);
      for( FVIter fv_it( mesh->fv_iter( FH(i))); fv_it.is_valid(); ++fv_it)
        points[j++] = mesh->point(*fv_it);
      Vec3 avgnorm(0.,0.,0.);
      for( int j = 0; j < val; ++j)
      {
        Vec3 va( (points[(j+1)%val] - points[j]));
        Vec3 vb( points[(j+val-1)%val] - points[j]);
        if( (va.normalize() + vb.normalize()).norm() > 1e-5)
        {
          normals[j] = (va%vb).normalize();
          avgnorm += normals[j];
        }
        else
          normals[j] =Vec3(0.,0.,0.);
      }

      avgnorm.normalize();
      Scalar bestdotp(0.);
      Vec3 bestnorm(0.,0.,0.);
      for( int j = 0; j < val; ++j)
      {
        Scalar dotp(normals[j] | avgnorm);
        if( dotp > bestdotp)
        {
          bestnorm = normals[j];
          bestdotp = dotp;
        }
      }
      mesh->set_normal( FH(i), bestnorm);
    }
    (*o_it)->update();

    emit updatedObject( meshid, UPDATE_ALL);
    //
    // number of parts and measurements
    Scalar reallenofB1(0.121); // ca 12 cm
    Scalar conversionfactor(reallenofB1/global_strut_data()[BLUE].len(1)); // ca 12 cm

    Vec3 bbmin,bbmax,bb;
    dynamic_cast< PolyMeshObject* >( *o_it)->boundingBox( bbmin, bbmax);

    bb = (bbmax-bbmin);
    bb *= conversionfactor;

    tool_->info_te->clear();
    QString infotext("Measurements (m) : " + 
        QString::number(bb[0], 'g', 2) + " x " +
        QString::number(bb[1], 'g', 2) + " x " + 
        QString::number(bb[2], 'g', 2));
    infotext += "\nParts :\n" + 
      QString("  Nodes \t = ") + QString::number(mesh->n_vertices()) + QString("\n") +
      "  Struts \t = " + QString::number(mesh->n_edges()) + "\n" +
      "    Red (0/1/2/tot) \t = " + 
      QString::number(colorcounters[RED][0]) + "/" +
      QString::number(colorcounters[RED][1]) + "/" +
      QString::number(colorcounters[RED][2]) + "/" +
      QString::number(colorcounterssum[RED]) + "\n" +
      "    Blue (0/1/2/tot) \t = " + 
      QString::number(colorcounters[BLUE][0]) + "/" +
      QString::number(colorcounters[BLUE][1]) + "/" +
      QString::number(colorcounters[BLUE][2]) + "/" +
      QString::number(colorcounterssum[BLUE]) + "\n" +
      "    Yellow (0/1/2/tot) \t = " + 
      QString::number(colorcounters[YELLOW][0]) + "/" +
      QString::number(colorcounters[YELLOW][1]) + "/" +
      QString::number(colorcounters[YELLOW][2]) + "/" +
      QString::number(colorcounterssum[YELLOW]) + "\n";

    tool_->info_te->setPlainText(infotext);
    emit updateView();
  }
}

////////////////////////////////////////////////////////

void ZomeMeshViewerPlugin::slot_export_vzome( )
{
  for ( PluginFunctions::ObjectIterator o_it(PluginFunctions::TARGET_OBJECTS,DATA_POLY_MESH) ; o_it != PluginFunctions::objectsEnd(); ++o_it) 
  {
    POD* pod( get_POD( *o_it));
    if( !pod->iszomemesh())
      continue;

    QString fileName = QFileDialog::getSaveFileName(NULL, tr("Export vZome"),
        QDir::homePath() + "/data/ZomeMeshes/", tr("vZome (*.vef)"));
    QFileInfo fi( fileName);


    if( fileName == "")
    {
      std::cerr << __FUNCTION__ << ":: No file selected, export aborted!" << std::endl;
      return;
    }

    std::string completefilewithoutending( (fi.absolutePath() + "/"+ fi.baseName()).toStdString());

    pod->zmm().export_vzome( completefilewithoutending);

  }
}


////////////////////////////////////////////////////////

#if QT_VERSION < 0x050000
Q_EXPORT_PLUGIN2( zomemeshviewerplugin , ZomeMeshViewerPlugin );
#endif

