#include "ZomeMetaMeshLight.hh"
#include <fstream>
#include <iomanip>


#include <Poco/Exception.h>
#include <Poco/Zip/Decompress.h>
#include <Poco/DateTime.h>
#include <Poco/MemoryStream.h>
#include <Poco/StreamCopier.h>
#include <Poco/Zip/ZipArchive.h>
#include <Poco/Zip/Zip.h>
#include <Poco/Zip/ZipStream.h>
#include <iostream>
#include <fstream>

namespace ACG
{
  std::vector<std::string> & ZomeMetaMeshLight::split(const std::string &s, char delim, std::vector<std::string> &elems)
  {
    std::stringstream ss(s);
    std::string item;
    while (std::getline(ss, item, delim)) {
      elems.push_back(item);
    }
    return elems;
  }
  std::vector<std::string> ZomeMetaMeshLight::split(const std::string &s, char delim) {
    std::vector<std::string> elems;
    split(s, delim, elems);
    return elems;
  }

////////////////////////////////////////////////////////

Vec3 ZomeMetaMeshLight::transform_to_real( const Zome6dIntPos& _pos6)
{
  Scalar totalscale( scale_ / (2. * golden));

  Vec3 transfpos(0.,0.,0.);
  for( int c = 0; c < 3; ++c)
    transfpos[c] = (_pos6[c*2+0] * golden + _pos6[c*2+1]) * totalscale + origin_[c];
  return transfpos;
}

////////////////////////////////////////////////////////

void ZomeMetaMeshLight::transform_to_real( )
{
  for( VIter v_it = metamesh_.vertices_begin(); v_it != metamesh_.vertices_end(); ++v_it)
  {
    if( metamesh_.status(*v_it).deleted())
      continue;

    metamesh_.set_point( *v_it, transform_to_real( pos( v_it->idx())));
  }
}


////////////////////////////////////////////////////////

void ZomeMetaMeshLight::get_oriented_edge_end_vertices( const EH& _eh, VH& _vhi, VH& _vhj) const
{
  _vhj = metamesh_.to_vertex_handle( metamesh_.halfedge_handle( _eh, 0));
  _vhi = metamesh_.to_vertex_handle( metamesh_.halfedge_handle( _eh, 1));
}

////////////////////////////////////////////////////////

void ZomeMetaMeshLight::get_oriented_edge_halfedge( const EH& _eh, HEH& _heh) const
{
  _heh = metamesh_.halfedge_handle( _eh, 0);
}

///////////////////////////////////////////////////////

void ZomeMetaMeshLight::setup_nodehole_bits( )
{
  for( VIter v_it = metamesh_.vertices_begin(); v_it != metamesh_.vertices_end(); ++v_it)
    if( !metamesh_.status( *v_it).deleted())
      hole( *v_it).reset();

  for( EIter e_it = metamesh_.edges_begin(); e_it != metamesh_.edges_end(); ++e_it)
  {
    EH eh( *e_it);
    if( metamesh_.status(eh).deleted())
      continue;

    HEH h0( metamesh_.halfedge_handle(eh, 0));
    VH  v0( metamesh_.to_vertex_handle( h0));
    HEH h1( metamesh_.halfedge_handle(eh, 1));
    VH  v1( metamesh_.to_vertex_handle( h1));
    int dir0( dirs_.map_to_base_range( dirid( h1))); // from h1 to v1 is outgoing of v0
    int dir1( dirs_.map_to_base_range( dirid( h0)));
    if( hole( v0, dir0))
      std::cerr << " WARNING : " << v0 << " already has hole " << dir0 << std::endl;
    if( hole( v1, dir1))
      std::cerr << " WARNING : " << v1 << " already has hole " << dir1 << std::endl;
    hole(v0,dir0) = 1;
    hole(v1,dir1) = 1;
  }
  std::cerr << __FILE__ << " " << __FUNCTION__ << " done ... " << std::endl;
}

////////////////////////////////////////////////////////

void ZomeMetaMeshLight::load( std::string _filenamewoending, int _type)
{
  if( _type == 1) // filename is reference to .zm2 file, use name to load both a .zm2 and .off
    load_twoparts( _filenamewoending);
  else if( _type == 2) // filename is a reference to .zmc, a zip container containing both a .zm2 and .off
    load_zipparts( _filenamewoending);
  else
  {
    std::cerr << __FUNCTION__ << ":: UNKNOWN type " << std::endl;
    return;
  }

}

////////////////////////////////////////////////////////

void ZomeMetaMeshLight::load_twoparts( std::string _filenamewoending)
{
  std::cerr << __FILE__ << " " << __FUNCTION__ << std::endl;
  std::string zomefilename( _filenamewoending + ".zm2");
  std::string meshfilename( _filenamewoending + ".off");

  std::cerr << " ... cleaning mesh prior to load... " << std::endl;
  metamesh_.clean();
  std::cerr << " ... reading mesh " << meshfilename << std::endl;
  OpenMesh::IO::Options opt;
  if( !OpenMesh::IO::read_mesh( metamesh_, meshfilename, opt, false))
    std::cerr << "\n\n\t\t FAILED!\n\n " << std::endl;

  std::cerr << " ... loading zome data " << zomefilename << std::endl;

  std::ifstream loadfile( zomefilename.c_str());

  std::string header;
  {
    loadfile >> header;
    if( header == "pos")
      for( size_t i = 0; i < metamesh_.n_vertices(); ++i)
        loadfile >> pos(i);
  }
  {
    loadfile >> header;
    if( header == "realpos")
      for( size_t i = 0; i < metamesh_.n_vertices(); ++i)
      {
        Vec3 currpos(0,0,0);
        loadfile >> currpos;
        metamesh_.set_point( VH(i), currpos);
      }
  }
  {
    loadfile >> header;
    if( header == "scale")
      loadfile >> scale_;
  }
  {
    loadfile >> header;
    if( header == "origin")
      loadfile >> origin_;
  }
  {
    loadfile >> header;
    if( header == "edgedirids" )
    {
      // the loop only uses the number of edges, has nothing todo with the acutal edge i
      for( size_t i = 0; i < metamesh_.n_edges(); ++i)
      {
        int idi(-1), idj(-1);
        loadfile >> idi;
        loadfile >> idj;
        size_t curdir(0);
        loadfile >> curdir;
        for( VOHIter voh_it( metamesh_.voh_iter( VH(idi))); voh_it.is_valid(); ++voh_it)
        {
          VH tovh( metamesh_.to_vertex_handle( *voh_it));
          EH eh( metamesh_.edge_handle( *voh_it));
          if( tovh.idx() == idj)
          {
            VH vhi(-1), vhj(-1);
            get_oriented_edge_end_vertices( eh, vhi, vhj);
            if( vhi.idx() == idi)
              dirid( eh) = curdir;
            else
              dirid( eh) = dirs_.oppid(curdir);
          }
        }
      }
    }
  }

  std::cerr << " ... setting used holes for nodes ..." << std::endl;
  setup_nodehole_bits( );

  loadfile.close();
  std::cerr << " ... load DONE " << std::endl;
}

////////////////////////////////////////////////////////

void ZomeMetaMeshLight::load_zipparts( std::string _filenamewoending)
{
  std::cerr << __FILE__ << " " << __FUNCTION__ << std::endl;
  std::string zomefilename( _filenamewoending + ".zm2");
  std::string meshfilename( _filenamewoending + ".off");

  std::string zipfilename( _filenamewoending + ".zmc"); // zomemeshcollection

  // POCO Zip (w/o error checks)

  std::ifstream inp(zipfilename.c_str(), std::ios::binary);
  poco_assert (inp);
  Poco::Zip::ZipArchive arch(inp);
  std::vector<std::string> path = split( zomefilename, '/');
  const char *namezome = path.back().c_str();
  Poco::Zip::ZipArchive::FileHeaders::const_iterator it = arch.findHeader(namezome);
  poco_assert (it != arch.headerEnd());
  Poco::Zip::ZipInputStream zipinzome (inp, it->second);

  path = split( meshfilename, '/');
  const char *namemesh = path.back().c_str();
  it = arch.findHeader(namemesh);
  poco_assert (it != arch.headerEnd());
  Poco::Zip::ZipInputStream zipinmesh (inp, it->second);

  std::cerr << " ... cleaning mesh prior to load... " << std::endl;
  metamesh_.clean();
  std::cerr << " ... reading mesh " << meshfilename << std::endl;
  OpenMesh::IO::Options opt;
  if( !OpenMesh::IO::read_mesh( metamesh_, zipinmesh, ".OFF", opt, false))
    std::cerr << "\n\n\t\t FAILED!\n\n " << std::endl;

  std::cerr << " ... loading zome data " << zomefilename << std::endl;

  std::string header;
  {
    zipinzome >> header;
    if( header == "pos")
      for( size_t i = 0; i < metamesh_.n_vertices(); ++i)
        zipinzome >> pos(i);
  }
  {
    zipinzome >> header;
    if( header == "realpos")
      for( size_t i = 0; i < metamesh_.n_vertices(); ++i)
      {
        Vec3 currpos(0,0,0);
        zipinzome >> currpos;
        metamesh_.set_point( VH(i), currpos);
      }
  }
  {
    zipinzome >> header;
    if( header == "scale")
      zipinzome >> scale_;
  }
  {
    zipinzome >> header;
    if( header == "origin")
      zipinzome >> origin_;
  }
  {
    zipinzome >> header;
    if( header == "edgedirids" )
    {
      // the loop only uses the number of edges, has nothing todo with the acutal edge i
      for( size_t i = 0; i < metamesh_.n_edges(); ++i)
      {
        int idi(-1), idj(-1);
        zipinzome >> idi;
        zipinzome >> idj;
        size_t curdir(0);
        zipinzome >> curdir;
        for( VOHIter voh_it( metamesh_.voh_iter( VH(idi))); voh_it.is_valid(); ++voh_it)
        {
          VH tovh( metamesh_.to_vertex_handle( *voh_it));
          EH eh( metamesh_.edge_handle( *voh_it));
          if( tovh.idx() == idj)
          {
            VH vhi(-1), vhj(-1);
            get_oriented_edge_end_vertices( eh, vhi, vhj);
            if( vhi.idx() == idi)
              dirid( eh) = curdir;
            else
              dirid( eh) = dirs_.oppid(curdir);
          }
        }
      }
    }
  }

  std::cerr << " ... setting used holes for nodes ..." << std::endl;
  setup_nodehole_bits( );

  std::cerr << " ... load DONE " << std::endl;
}

////////////////////////////////////////////////////////

void ZomeMetaMeshLight::export_vzome( std::string _filenamewoending) 
{
  std::cerr << __FILE__ << " " << __FUNCTION__ << std::endl;

  std::string vzomefilename( _filenamewoending + ".vef");

  std::cerr << " ... writing " << vzomefilename << std::endl;

  std::cerr << " ... saving zome data ..." << std::endl;
  std::ofstream savefile( vzomefilename.c_str());
  savefile << std::setprecision(15);

  // write version
  savefile << std::string("vZome VEF 6") << std::endl;
  
  // write field
  savefile << std::string("field golden") << std::endl;

  // write nodes VEF (a+g*b) vs ZM2 (a*g+b)
  savefile << metamesh_.n_vertices() << std::endl;

  for( size_t i = 0; i < metamesh_.n_vertices(); ++i)
  {
    savefile << "(0,0)";
    // divide by 2gamma
    for( int c = 0; c < 3; ++c)
      //savefile << "(" << (pos(i)[2*c+1]-pos(i)[2*c+0])<< "/2," << (2*pos(i)[2*c+0]-pos(i)[2*c+1])  << "/2) ";
      // WORKS
      //savefile << "(" << (pos(i)[2*c+0]) << "/2," << pos(i)[2*c+1] << "/2) ";
      // SCALED BY 1/gamma^3
      savefile << " (" <<  (2*pos(i)[2*c+1]-pos(i)[2*c+0])<< "," << (2*pos(i)[2*c+0]-3*pos(i)[2*c+1]) << ")";
    savefile << std::endl;
  }

  // write struts
  savefile << metamesh_.n_edges() << std::endl;

  for( size_t i = 0; i < metamesh_.n_edges(); ++i)
    savefile << metamesh_.to_vertex_handle(metamesh_.halfedge_handle( EH(i), 0)).idx() << " " << metamesh_.to_vertex_handle( metamesh_.halfedge_handle( EH(i), 1)).idx() << std::endl;
  //savefile << std::endl;
  
  // write faces
  savefile << metamesh_.n_faces() << std::endl;

  for( size_t i = 0; i < metamesh_.n_faces(); ++i)
  {
    savefile << metamesh_.valence( FH(i)) ;
    for( FVIter fv_it( metamesh_.fv_iter( FH(i))); fv_it.is_valid(); ++fv_it)
      savefile << " " << fv_it->idx();
    savefile << std::endl;
  }

  savefile.close();
  std::cerr << " ... export DONE " << std::endl;
}

////////////////////////////////////////////////////////
///// Initialization
////////////////////////////////////////////////////////

////////////////////////////////////////////////////////

void ZomeMetaMeshLight::add_properties( )
{
  if( !metamesh_.get_property_handle( metanodeposprop_, "metanodeposprop_"))
    metamesh_.add_property( metanodeposprop_, "metanodeposprop_");

  if( !metamesh_.get_property_handle( edgediridprop_, "edgediridprop_"))
    metamesh_.add_property( edgediridprop_, "edgediridprop_");

  if( !metamesh_.get_property_handle( propv_nodeholes_, "propv_nodeholes_"))
    metamesh_.add_property( propv_nodeholes_, "propv_nodeholes_");
}

////////////////////////////////////////////////////////

void ZomeMetaMeshLight::rem_properties( )
{
  if( metamesh_.get_property_handle( metanodeposprop_, "metanodeposprop_"))
    metamesh_.remove_property( metanodeposprop_);

  if( metamesh_.get_property_handle( edgediridprop_, "edgediridprop_"))
    metamesh_.remove_property( edgediridprop_);

  if( metamesh_.get_property_handle( propv_nodeholes_, "propv_nodeholes_"))
    metamesh_.remove_property( propv_nodeholes_);

}

}

