////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2011, Computer Graphics Group RWTH Aachen University         //
// All rights reserved.                                                       //
////////////////////////////////////////////////////////////////////////////////

// Cf. http://developer.download.nvidia.com/assets/gamedev/files/sdk/11/FXAA_WhitePaper.pdf

#version 150
#pragma optionNV unroll all

uniform sampler2DRect uSamplerInput;

out vec4 fOut;

const float cEdgeThreshold = 1.0 / 8.0;
const float cEdgeThresholdMin = 1.0 / 16.0;
const float cSubpixelTrim = 1.0 / 4.0;
const float cSubpixelTrimScale = 8.0;
const float cSubpixelCap = 3.0 / 4.0;
const int cSearchSteps = 8;

float luma(in vec3 iColor)
{
    // Neglect blue channel, so this is just one MAD
    return iColor.g * (0.587 / 0.299) + iColor.r;
}

void main(void)
{
    fOut = texture(uSamplerInput, gl_FragCoord.xy);

    vec3 colorN = texture(uSamplerInput, gl_FragCoord.xy + vec2( 0,-1)).rgb;
    vec3 colorW = texture(uSamplerInput, gl_FragCoord.xy + vec2(-1, 0)).rgb;
    vec3 colorM = fOut.rgb; //texture(uSamplerInput, gl_FragCoord.xy + vec2( 0, 0)).rgb;
    vec3 colorE = texture(uSamplerInput, gl_FragCoord.xy + vec2( 1, 0)).rgb;
    vec3 colorS = texture(uSamplerInput, gl_FragCoord.xy + vec2( 0, 1)).rgb;


    // Local Contrast Check
    float lumaN = luma(colorN);
    float lumaW = luma(colorW);
    float lumaM = luma(colorM);
    float lumaE = luma(colorE);
    float lumaS = luma(colorS);

    float rangeMin = min(lumaN, min(min(lumaW, lumaM), min(lumaE, lumaS)));
    float rangeMax = max(lumaN, max(max(lumaW, lumaM), max(lumaE, lumaS)));
    float range = rangeMax - rangeMin;

    if(range >= max(cEdgeThresholdMin, rangeMax * cEdgeThreshold))
    {
        // Edge detected, perform FXAA

        // Estimate local contrast
        float lumaLowpass = (lumaN + lumaW + lumaE + lumaS) * 0.25;
        float rangeLowpass = abs(lumaM - lumaLowpass);
        float blendLowpass = max(0.0, (rangeLowpass / range) - cSubpixelTrim) * cSubpixelTrimScale;
        blendLowpass = min(blendLowpass, cSubpixelCap);

        // Create color lowpass value using a 3x3 box filter
        vec3 colorLowpass = colorN + colorW + colorM + colorE + colorS;
        vec3 colorNW = texture(uSamplerInput, gl_FragCoord.xy + vec2(-1,-1)).rgb;
        vec3 colorNE = texture(uSamplerInput, gl_FragCoord.xy + vec2( 1,-1)).rgb;
        vec3 colorSW = texture(uSamplerInput, gl_FragCoord.xy + vec2(-1, 1)).rgb;
        vec3 colorSE = texture(uSamplerInput, gl_FragCoord.xy + vec2( 1, 1)).rgb;
        colorLowpass += (colorNW + colorNE + colorSW + colorSE);
        colorLowpass *= 1.0/9.0;

        // Find out whether the current edge is horizontal/vertical
        float lumaNW = luma(colorNW);
        float lumaNE = luma(colorNE);
        float lumaSW = luma(colorSW);
        float lumaSE = luma(colorSE);

        float verticalEdge = abs( 0.25*lumaNW - 0.5*lumaN + 0.25*lumaNE ) +
                             abs( 0.5 *lumaW  - 1.0*lumaM + 0.5 *lumaE  ) +
                             abs( 0.25*lumaSW - 0.5*lumaS + 0.25*lumaSE );

        float horizontalEdge = abs( 0.25*lumaNW - 0.5*lumaW + 0.25*lumaSW ) +
                               abs( 0.5 *lumaN  - 1.0*lumaM + 0.5 *lumaS  ) +
                               abs( 0.25*lumaNE - 0.5*lumaE + 0.25*lumaSE );

        bool isHorizontalEdge = (horizontalEdge >= verticalEdge);

        if(!isHorizontalEdge) lumaN = lumaW;
        if(!isHorizontalEdge) lumaS = lumaE;

        float gradientN = abs(lumaN - lumaM);
        float gradientS = abs(lumaS - lumaM);

        lumaN = (lumaN + lumaM) * 0.5;
        lumaS = (lumaS + lumaM) * 0.5;

        bool pairN = gradientN >= gradientS;
        float lengthSign = -1.0;
        if(!pairN)
        {
            lumaN = lumaS;
            gradientN = gradientS;
            lengthSign *= -1.0;
        }

        vec2 posN;
        posN.x = gl_FragCoord.x + (isHorizontalEdge ? 0.0 : lengthSign*0.5);
        posN.y = gl_FragCoord.y + (isHorizontalEdge ? lengthSign*0.5 : 0.0);

        gradientN *= 0.25;

        vec2 posP = posN;
        vec2 offNP = isHorizontalEdge ? vec2(1.0, 0.0) : vec2(0.0, 1.0);
        float lumaEndN = lumaN;
        float lumaEndP = lumaN;
        bool doneN = false;
        bool doneP = false;
        posN += offNP * vec2(-1.0,-1.0);
        posP += offNP * vec2( 1.0, 1.0);

        // Here be dragons
        for(int i = 0; i < cSearchSteps; ++i)
        {
            if(!doneN) lumaEndN = luma(texture(uSamplerInput, posN).rgb);
            if(!doneP) lumaEndP = luma(texture(uSamplerInput, posP).rgb);

            doneN = doneN || (abs(lumaEndN - lumaN) >= gradientN);
            doneP = doneP || (abs(lumaEndP - lumaN) >= gradientN);

            if(doneN && doneP)
                break;

            if(!doneN) posN -= offNP;
            if(!doneP) posP += offNP;
        }

        float dstN = isHorizontalEdge ? gl_FragCoord.x - posN.x : gl_FragCoord.y - posN.y;
        float dstP = isHorizontalEdge ? posP.x - gl_FragCoord.x : posP.y - gl_FragCoord.y;
        bool directionN = dstN < dstP;

        lumaEndN = directionN ? lumaEndN : lumaEndP;

        if((lumaM - lumaN < 0.0) == (lumaEndN - lumaN < 0.0)) lengthSign = 0.0;
        float spanLength = (dstP + dstN);
        dstN = directionN ? dstN : dstP;
        float subPixelOffset = (0.5 + (dstN * (-1.0 / spanLength))) * lengthSign;

        vec3 outColor = texture(uSamplerInput, vec2(
            gl_FragCoord.x + (isHorizontalEdge ? 0.0 : subPixelOffset),
            gl_FragCoord.y + (isHorizontalEdge ? subPixelOffset : 0.0)
        )).xyz;

        // Blend in the low-passed color value
        fOut.rgb = mix(outColor, colorLowpass, blendLowpass);
    }
    else
    {
        // No FXAA
    }
}
