////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2011, Computer Graphics Group RWTH Aachen University         //
// All rights reserved.                                                       //
////////////////////////////////////////////////////////////////////////////////

#version 150
uniform vec3 uFirstColor;
uniform vec3 uSecondColor;

uniform samplerCube uSamplerSky;

uniform vec3  uPlanetPos;
uniform float uPlanetRadius;
uniform mat4  uViewMatrixInverse;
uniform mat4  uProjectionMatrixInverse;


const vec3 atmosphereColorZenit = vec3(0.05,0.25,0.4);
const vec3 atmosphereColorGround = vec3(0.5,0.8,1.0);

uniform float uAtmosphereHeight; // absolute, should be > uPlanetRadius
const float atmosphereDensity = 0.06; // how much light will be absorbed when travelling trough 1 unit on the ground air?
const float atmosphereLightOverhang = 2.75; // how far the atmosphere is light although the ground is already dark (looks more realistic, kinda)


in vec2 vTexCoordScreen;
in vec3 vSkyDir;
in vec3 vVelocity;

out vec4 fFragDataBlack0;
out vec4 fFragDataBlack1;
out vec4 fFragDataGradient;
out vec4 fFragDataVelocity;

const float pi = 3.1415926535;

struct SphereIntersection
{
    vec3 near;
    vec3 far;
    bool intersects;
};

//calculates the intersections of a given ray and a sphere located at (0,0,0)
SphereIntersection intersectSphere(vec3 pos, vec3 dir, float sphereRadius)
{
    //This algorithm is basically copypasted from "Real-Time Rendering" by Akenine and Haines
    //I have no idea what it actually does, but it works.

    float r = sphereRadius*sphereRadius;
    float s = dot(-pos, dir);
    float l = dot(pos, pos);
    float m = l - s*s;

    //no intersection at all?
    if((s<0 && l > r) || m > r)
    {
        SphereIntersection si;
        si.intersects = false;
        return si;
    }

    float q = sqrt(r - m);

    float t1;
    float t2;
    t1=max(s-q,0);
    t2 =max(s+q,0);

    SphereIntersection si;
    si.intersects = true;
    si.near = pos + min(t1,t2)*dir;
    if(min(t1,t2)<0)
        si.near = pos;
    si.far = pos + max(t1,t2)*dir;
    return si;
}

//build 3d-texcoords with latitude/longitude and distance to center as third dimension
vec3 getUV(vec3 pos)
{
    float latitude = atan(pos.x, pos.z);
    float longitude = asin(pos.y/uPlanetRadius);

    //build 3d-texcoords with distance to center as third dimension
    return vec3(latitude/pi, longitude/pi, length(pos)/(uPlanetRadius * 4.0)) * 0.5 + 0.5;
}

// Returns atmosphere color, given that a ray passes through the atmosphere only (not the planet's surface)
vec4 atmosphereColor(float distance, float height)
{
    //this is a simple faked fog, no real scattering (lighting is not taken into account).
    float h = 1-(height - uPlanetRadius)/(uAtmosphereHeight - uPlanetRadius);


    //the fog's intensity is given by the distance travelled trough the atmosphere,
    //assuming the fog has the density of the average height at each point.
    //the density just linearly drops from atmosphereDensity to zero from planetSize to uAtmosphereHeight.
    float a = clamp((atmosphereDensity * distance)*h,0.0,1.0);
    return vec4(mix(atmosphereColorZenit, atmosphereColorGround, h), a);
}

float getLight(vec3 n) {
    return dot(normalize(n), vec3(-1,0,0))*0.8;
}

void main (void)
{
    fFragDataVelocity = vec4(vVelocity,1.0);
    fFragDataBlack0   = vec4(0.0);
    fFragDataBlack1   = vec4(0.0);

    //Vertex position [at a pixel] from NDC [-1..1]
    vec4 ndc = vec4(vTexCoordScreen.x, vTexCoordScreen.y, 0.0, 1.0) * 2.0 - 1.0;
    vec4 eyeDir = uProjectionMatrixInverse * ndc;
    eyeDir /= eyeDir.w;

    vec4 worldDir = uViewMatrixInverse * vec4(eyeDir.xyz, 0.0);
    worldDir.xyz = normalize(worldDir.xyz);

    vec4 camPos = uViewMatrixInverse * vec4(0.0, 0.0, 0.0, 1.0);


    vec4 color = vec4(0.0);

    SphereIntersection atmosIntersection = intersectSphere(camPos.xyz - uPlanetPos, worldDir.xyz, uAtmosphereHeight);

    if(atmosIntersection.intersects)
    {
        float lowestHeight = min(length((atmosIntersection.near+atmosIntersection.far)*0.5),
                                 length(atmosIntersection.near - uPlanetPos)
                                );

        vec4 atmosColor = atmosphereColor(length(atmosIntersection.near - atmosIntersection.far), lowestHeight);


        SphereIntersection planetIntersection = intersectSphere(camPos.xyz - uPlanetPos, worldDir.xyz, uPlanetRadius);

        if(planetIntersection.intersects)
        {

            color.rgb = mix(vec3(0.2, 0.5, 0.8), atmosColor.rgb, atmosColor.a) * vec3(getLight(planetIntersection.near));
            color.a = 1.0;
            //gl_FragCoord.z = length(uViewMatrix[3].xyz - intersection.near);/
        }
        else
        {
            color = vec4(atmosColor.rgb * getLight((atmosIntersection.near+atmosIntersection.far) * 0.5) * atmosColor.a, atmosColor.a);
            color += texture(uSamplerSky, normalize(vSkyDir)) * clamp(1.0 - color.a, 0.1, 1.0);
            //gl_FragCoord.z = 1.0/0.0;
        }
    }
    else
    {
        color += texture(uSamplerSky, normalize(vSkyDir));
        //gl_FragCoord.z = 1.0/0.0;
    }

    fFragDataGradient = color;
}
