////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2011, Computer Graphics Group RWTH Aachen University         //
// All rights reserved.                                                       //
////////////////////////////////////////////////////////////////////////////////

#version 150
uniform sampler2D uSamplerTextureOuter;
uniform sampler2D uSamplerTextureInner;
uniform samplerCube uSamplerSky;
uniform mat4 uPreviousModelViewProjectionMatrix;
uniform mat4 uModelViewProjectionMatrix;
uniform mat4 uViewMatrix;
uniform mat3 uInvNormalMat;

uniform float uTime;

//uniforms for CoC computation
uniform float uAperture;
uniform float uFocalLength;
uniform float uFocus;
uniform vec2 uCameraRange;

in vec3 vEyeNormal;
in vec3 vEyeTangent;
in vec3 vEyeBitangent;
in vec3 vWorldPos;
in vec2 vVelocity;
in vec4 eyePos;

out vec4 fFragDataEyeNormal;
out vec4 fFragDataColor;
out vec4 fFragDataVelocity;


void main(void)
{
    // normalize tangents
    vec3 eyeNormal = normalize(vEyeNormal);
    vec3 eyeTangent = normalize(vEyeTangent);
    vec3 eyeBitangent = normalize(vEyeBitangent);

    mat3 normalTransform = mat3(eyeTangent, eyeBitangent, eyeNormal);  //from tangentspace to eyespace

    vec3 newEyeNormal = eyeNormal;//normalize(normalTransform * (texture(uSamplerNormalTexture, vTexCoord).rgb*2.0-1.0));

    //calculate the texcoords for the skybox (the same that would also be computed by the clear shader
    vec3 worldNormal = uInvNormalMat * eyeNormal.xyz;
    vec3 skyDir = uInvNormalMat * normalize(eyePos.xyz);

    //for our warp effect, we calculate texcoords based on the angle on the xy-plane
    //and the distance to the z-axis
    float l = length(vWorldPos.xy);

    //the wormhole consists of two tubes and a warping zone.
    //we distinct them by their distance to the z-axis.
    //warping zone is between l=0.5 and l=3.5
    //the distortion should only occur on the warping zone, so we compute the
    //distortion factor in a way that it equals zero at l=0.5 and 3.5 and equals
    //one at l=1.5
    float innerDistortion = (l - 0.5) / (1.5 - 0.5);
    float outerDistortion = 1.0 - (l - 1.5) / (3.5 - 1.5);

    float distortion = clamp(min(innerDistortion, outerDistortion), 0.0, 1.0);
    distortion = distortion*distortion;

    //use the inverted inner distortion for the intensity of the inner texture
    //this way, the texture will blend in from l = 1.5 to l=0.5
    float innerIntensity = clamp(1.0 - innerDistortion, 0.0, 1.0);

    //we use two texture coordinates with different rotations,
    //for both the inner and outer textures are mixed

    //add l to the angle for a spiral-like effect and add the z position to the y coord
    //to get texcoords on the inner tube also.
    vec2 texCoord1  = vec2(atan(vWorldPos.x/vWorldPos.y)+l, l + vWorldPos.z) + vec2(0.0, 1.0)*(uTime*1.5+1.0);
    vec2 texCoord2  = vec2(atan(0.5 + vWorldPos.x/vWorldPos.y)+l, l + vWorldPos.z) + vec2(0.0, 1.0)*uTime;

    vec4 texOuter = texture(uSamplerTextureOuter, texCoord1);
    vec4 texInner = texture(uSamplerTextureInner, texCoord1);

    vec4 tex1 = clamp(texOuter*distortion + texInner*innerIntensity, vec4(0.0), vec4(1.0));

    texOuter = texture(uSamplerTextureOuter, texCoord2);
    texInner = texture(uSamplerTextureInner, texCoord2);

    vec4 tex2 = clamp(texOuter*distortion + texInner*innerIntensity, vec4(0.0), vec4(1.0));

    //the distortion is also influenced by an animated texture
    texCoord1  = vec2(atan(vWorldPos.x/vWorldPos.y)+l, l + vWorldPos.z) + vec2(0.0, 1.0)*(uTime*0.1);
    float distortTexture = texture(uSamplerTextureInner, texCoord1).b*0.5;

    //now do the actual distortion: blend between the correct sky dir and the negative normal
    //based on the distortion factors
    skyDir = normalize(mix(skyDir, -worldNormal, 0.3*(distortion+distortion*distortTexture)));

    //based on that distorted 3d-coords, get the sky color
    vec4 sky = texture(uSamplerSky, skyDir);

    //and now blend everything together
    fFragDataColor.rgb = (sky.rgb * (1.0 - innerIntensity)) + pow((tex1.rgb+tex2.rgb)*0.7, vec3(2.5));

    fFragDataColor.a = 1.0;
    fFragDataEyeNormal.rgb = (newEyeNormal * 0.5 + 0.5);
    fFragDataEyeNormal.a   = 0.0;

    fFragDataVelocity.xy = vVelocity;
    fFragDataVelocity.z = 0.0f;
    fFragDataVelocity.a = -1.0;

    float objectDistance = -uCameraRange.y * uCameraRange.x / (gl_FragCoord.z * (uCameraRange.y - uCameraRange.x) - uCameraRange.y);
    fFragDataVelocity.a *= abs(uAperture * (uFocalLength * (objectDistance - uFocus)) / (objectDistance * (uFocus - uFocalLength)));

}
