////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2011, Computer Graphics Group RWTH Aachen University         //
// All rights reserved.                                                       //
////////////////////////////////////////////////////////////////////////////////

#version 150

#pragma ACGLimport "Camera.glsl"
#pragma optionNV unroll all

uniform sampler2D uSamplerSun;

in vec2  vScreenTexCoord;
in vec2  vScreenSunCoord; // Sun position in [0,1]x[0,1] screen coords
in float vSunIntensity;   // Sun intensity falls off if sun is outside of screen

out vec4 fColor;

const int NUM_SAMPLES = 16;
const float DENSITY = 0.75;
const float EXPOSURE = 1.0;
const float DECAY = 0.9;
const float WEIGHT = 1.0 / NUM_SAMPLES;

void main()
{
    // Calculate vector from pixel to light source in screen space.
    vec2 deltaTexCoord = vScreenTexCoord - vScreenSunCoord;

    // Divide by number of samples and scale by control factor.
    deltaTexCoord *= 1.0 / NUM_SAMPLES * DENSITY;

    // Store initial sample.
    vec3 color = vec3(0.0);

    // Set up illumination decay factor.
    float illuminationDecay = 1.0;

    // Evaluate summation from Equation 3 NUM_SAMPLES iterations.
    vec2 texCoord = vScreenTexCoord;
    for(int i = 0; i < NUM_SAMPLES; i++)
    {
        // Step sample location along ray.
        texCoord -= deltaTexCoord;

        // Retrieve sample at new location.
        vec3 sunSample = texture(uSamplerSun, texCoord).rgb;

        // Apply sample attenuation scale/decay factors.
        sunSample *= illuminationDecay * WEIGHT;

        // Accumulate combined color.
        color += sunSample;

        // Update exponential decay factor.
        illuminationDecay *= DECAY;
    }

    // Output final color with a further scale control factor.
    fColor = vec4(color * EXPOSURE, 1.0);
}
